#ifndef GEOMETRY_H
#define GEOMETRY_H

#include "glow.hpp"
#ifdef _WIN32
#include <glm/gtc/constants.hpp>
const float PI = glm::pi<float>();
#else
const float PI = 3.14152f;
#endif

const bool TSTRIP = true;


struct VertexC {
    float x,y,z;
    float r,g,b,a;
};

namespace {
typedef std::shared_ptr<glow::Geometry<VertexC,GLint,TSTRIP>> GP;

}

inline void box(GP g, glm::vec3 a, glm::vec3 b, glm::vec4 color = {0,0,0,1}) {
    glm::vec3 tmp = glm::min(a,b);
    b=glm::max(a,b);
    a=tmp;
    g->addQuad( // Front
        {a.x, a.y, a.z,    color.r, color.g, color.b, color.a},
        {b.x, a.y, a.z,    color.r, color.g, color.b, color.a},
        {b.x, a.y, b.z,    color.r, color.g, color.b, color.a},
        {a.x, a.y, b.z,    color.r, color.g, color.b, color.a});
    g->addQuad( // Right
        {b.x, a.y, a.z,    color.r, color.g, color.b, color.a},
        {b.x, b.y, a.z,    color.r, color.g, color.b, color.a},
        {b.x, b.y, b.z,    color.r, color.g, color.b, color.a},
        {b.x, a.y, b.z,    color.r, color.g, color.b, color.a});
    g->addQuad( // Left
        {a.x, b.y, a.z,    color.r, color.g, color.b, color.a},
        {a.x, a.y, a.z,    color.r, color.g, color.b, color.a},
        {a.x, a.y, b.z,    color.r, color.g, color.b, color.a},
        {a.x, b.y, b.z,    color.r, color.g, color.b, color.a});
    g->addQuad( // Back
        {b.x, b.y, a.z,    color.r, color.g, color.b, color.a},
        {a.x, b.y, a.z,    color.r, color.g, color.b, color.a},
        {a.x, b.y, b.z,    color.r, color.g, color.b, color.a},
        {b.x, b.y, b.z,    color.r, color.g, color.b, color.a});
    g->addQuad( // Top
        {a.x, a.y, b.z,    color.r, color.g, color.b, color.a},
        {b.x, a.y, b.z,    color.r, color.g, color.b, color.a},
        {b.x, b.y, b.z,    color.r, color.g, color.b, color.a},
        {a.x, b.y, b.z,    color.r, color.g, color.b, color.a});
    g->addQuad( // Bottom
        {b.x, a.y, a.z,    color.r, color.g, color.b, color.a},
        {a.x, a.y, a.z,    color.r, color.g, color.b, color.a},
        {a.x, b.y, a.z,    color.r, color.g, color.b, color.a},
        {b.x, b.y, a.z,    color.r, color.g, color.b, color.a});
}

inline void pipe(GP g, glm::vec3 a, float ra, glm::vec3 b, float rb, int segments = 7, glm::vec4 color = {0,0,0,1}) {
    glm::vec3 c = glm::normalize(glm::cross(a,b-a));
    glm::vec3 d = glm::normalize(glm::cross(c,b-a));
    std::vector<VertexC> vv;
    // inside
    for (int i = 0; i<=segments; i++) {
        float phi = 2.f*PI*i/segments;
        float sinp = glm::sin(phi);
        float cosp = glm::cos(phi);
        glm::vec3 p1 = a+ra*c*sinp+ra*d*cosp;
        glm::vec3 p2 = b+rb*c*sinp+rb*d*cosp;
        vv.push_back(VertexC({p1.x,p1.y,p1.z,color.r,color.g,color.b,color.a}));
        vv.push_back(VertexC({p2.x,p2.y,p2.z,color.r,color.g,color.b,color.a}));
    }
    g->addTriStrip(vv);
    // outside
    for (int i = 0; i<=segments; i++) {
        float phi = 2.f*PI*i/segments;
        float sinp = glm::sin(phi);
        float cosp = glm::cos(phi);
        glm::vec3 p1 = a-ra*c*sinp+ra*d*cosp;
        glm::vec3 p2 = b-rb*c*sinp+rb*d*cosp;
        vv.push_back(VertexC({p1.x,p1.y,p1.z,color.r,color.g,color.b,color.a}));
        vv.push_back(VertexC({p2.x,p2.y,p2.z,color.r,color.g,color.b,color.a}));
    }
    g->addTriStrip(vv);
}

inline void cone(GP g, glm::vec3 a, glm::vec3 b, float r, int segments = 7, glm::vec4 color = {0,0,0,1}) {
    glm::vec3 c = glm::normalize(glm::cross(a,b-a));
    glm::vec3 d = glm::normalize(glm::cross(c,b-a));
    std::vector<VertexC> vv;
    // tip
    vv.push_back(VertexC({b.x,b.y,b.z,color.r,color.g,color.b,color.a}));
    // base
    for (int i = 0; i<segments; i++) {
        float phi = 2.f*PI*i/segments;
        float sinp = glm::sin(phi);
        float cosp = glm::cos(phi);
        glm::vec3 p = a+r*c*sinp+r*d*cosp;
        vv.push_back(VertexC({p.x,p.y,p.z,color.r,color.g,color.b,color.a}));
    }
    g->addTriFan(vv);
}

inline void wedge(GP g, glm::vec3 a, glm::vec3 b, glm::vec3 c, float width, glm::vec4 color = {0,0,0,1}) {
    glm::vec3 d = width*glm::normalize(glm::cross(c-a,b-a));
    g->addTriStrip(std::vector<VertexC>({
        {a.x,    a.y,    a.z,    color.r,color.g,color.b,color.a},
        {a.x+d.x,a.y+d.y,a.z+d.z,color.r,color.g,color.b,color.a},
        {b.x,    b.y,    b.z,    color.r,color.g,color.b,color.a},
        {b.x+d.x,b.y+d.y,b.z+d.z,color.r,color.g,color.b,color.a},
        {c.x,    c.y,    c.z,    color.r,color.g,color.b,color.a},
        {c.x+d.x,c.y+d.y,c.z+d.z,color.r,color.g,color.b,color.a},
        {a.x,    a.y,    a.z,    color.r,color.g,color.b,color.a},
        {a.x+d.x,a.y+d.y,a.z+d.z,color.r,color.g,color.b,color.a}
        }));

    g->addTri(
        {a.x,    a.y,    a.z,    color.r,color.g,color.b,color.a},
        {b.x,    b.y,    b.z,    color.r,color.g,color.b,color.a},
        {c.x,    c.y,    c.z,    color.r,color.g,color.b,color.a});

    g->addTri(
        {a.x+d.x,a.y+d.y,a.z+d.z,color.r,color.g,color.b,color.a},
        {c.x+d.x,c.y+d.y,c.z+d.z,color.r,color.g,color.b,color.a},
        {b.x+d.x,b.y+d.y,b.z+d.z,color.r,color.g,color.b,color.a});
}

inline void ball(GP g, glm::vec3 a, float r, int segments = 7, int slices = 4, glm::vec4 color = {0,0,0,1}) {
    static const float pi = PI;
    float theta_slice = pi/(slices+2);

    std::vector<VertexC> vv;
    float rt = r*glm::sin(theta_slice);
    float ht = a.z+r*glm::cos(theta_slice);
    vv.push_back(VertexC({a.x,a.y,a.z+r,color.r,color.g,color.b,color.a}));
    for (int seg = 0; seg<segments; seg++) {
        float phi = 2.f*pi*seg/segments;
        glm::vec2 p = glm::vec2(a)+glm::vec2(rt*glm::cos(phi),0)+glm::vec2(0,rt*glm::sin(phi));
        vv.push_back(VertexC({p.x,p.y,ht,color.r,color.g,color.b,color.a}));
    }
    g->addTriFan(vv);

    for (int slc = 0; slc < slices; slc++) {
        vv.clear();
        float r1 = r*glm::sin((slc+1)*theta_slice);
        float r2 = r*glm::sin((slc+2)*theta_slice);
        glm::vec3 h1 = a+glm::vec3(0,0,r)*glm::cos((slc+1)*theta_slice);
        glm::vec3 h2 = a+glm::vec3(0,0,r)*glm::cos((slc+2)*theta_slice);
        for (int seg = 0; seg<=segments; seg++) {
            float phi = 2.f*pi*seg/segments;
            float sinp = glm::sin(phi);
            float cosp = glm::cos(phi);
            glm::vec3 p1 = h1+glm::vec3(r1*cosp,0,0)+glm::vec3(0,r1*sinp,0);
            glm::vec3 p2 = h2+glm::vec3(r2*cosp,0,0)+glm::vec3(0,r2*sinp,0);
            vv.push_back(VertexC({p1.x,p1.y,p1.z,color.r,color.g,color.b,color.a}));
            vv.push_back(VertexC({p2.x,p2.y,p2.z,color.r,color.g,color.b,color.a}));
        }
        g->addTriStrip(vv);
    }

    vv.clear();
    ht = a.z-r*glm::cos(theta_slice);
    vv.push_back(VertexC({a.x,a.y,a.z-r,color.r,color.g,color.b,color.a}));
    for (int seg = segments-1; seg>=0; seg--) {
        float phi = 2.f*pi*seg/segments;
        glm::vec2 p = glm::vec2(a)+glm::vec2(rt*glm::cos(phi),0)+glm::vec2(0,rt*glm::sin(phi));
        vv.push_back(VertexC({p.x,p.y,ht,color.r,color.g,color.b,color.a}));
    }
    g->addTriFan(vv);

}


#endif // GEOMETRY_H
