#ifndef QDW_HPP
#define QDW_HPP

#include "sdw.hpp"

#include <sstream>

namespace qdw {

namespace {
using namespace sdw;

template <typename T> // Why isn't this part of std:: already?
class _reverse_range {
    T& x_;
public:
    _reverse_range (T& x): x_ (x) {}
    auto begin() const -> decltype(this->x_.rbegin()) {return x_.rbegin();}
    auto end () const -> decltype(this->x_.rend()) {return x_.rend();}
};

template <typename T>
_reverse_range<T> reverse(T& x) {return _reverse_range<T> (x);}
}

class Widget;
typedef std::function<void (Widget *caller, const SDL_Event &e)> Callback; // Works fine with lambdas
typedef std::function<void (Widget *active, Widget *passive, const SDL_Event &e)> Interaction;
class Gui;

class Theme {
public:
    Theme(Font _font, SDL_Color textcol) : font(_font), tilesize(0), bordersize(0) {color[TEXT_ENABLED] = textcol;}
    Theme(Surface _tileset, Font _font, int _tilesize = 16, int _bordersize = 2) throw(Error) :
        tileset(_tileset), font(_font), tilesize(_tilesize), bordersize(_bordersize) {
            if (tileset.width()<tilesize*16) throw Error("qdw::Theme.tileset must be at least 16 tiles wide.");
            if (tileset.height()<tilesize*16) throw Error("qdw::Theme.tileset must be at least 16 tiles high.");
            if (tilesize>0) {
                Uint32 transparent = tileset.pixel(tilesize*16-1,0);
                if (transparent&&AMASK) tileset.replaceColor(transparent,0);
                int &ts=tilesize;
                int ts1 = ts-1;
                int ts2 = ts/2;
                int ts15 = ts*15;
                int ts155 = ts15+ts2;
                int ts161 = 16*ts-1;
                color[WINDOW_BACKGROUND] = tileset.get_pixel(ts2,ts155); // We want SDL_Color from now on
                color[EDIT_BACKGROUND] = tileset.get_pixel(ts+ts2,ts155);
                color[TEXT_ENABLED] = tileset.get_pixel(2*ts+ts2,ts155);
                color[TEXT_DISABLED] = tileset.get_pixel(3*ts+ts2,ts155);
                color[TEXT_SELECTED] = tileset.get_pixel(4*ts+ts2,ts155);
                color[TEXT_SELECTION] = tileset.get_pixel(5*ts+ts2,ts155);
                color[TITLE_ENABLED_TEXT] = tileset.get_pixel(6*ts+ts2,ts155);
                color[TITLE_DISABLED_TEXT] = tileset.get_pixel(7*ts+ts2,ts155);
                color[TITLE_ENABLED_TL] = tileset.get_pixel(8*ts,ts15);
                color[TITLE_ENABLED_TR] = tileset.get_pixel(9*ts+ts1,ts15);
                color[TITLE_ENABLED_BL] = tileset.get_pixel(8*ts,ts161);
                color[TITLE_ENABLED_BR] = tileset.get_pixel(9*ts+ts1,ts161);
                color[TITLE_DISABLED_TL] = tileset.get_pixel(10*ts,ts15);
                color[TITLE_DISABLED_TR] = tileset.get_pixel(11*ts+ts1,ts15);
                color[TITLE_DISABLED_BL] = tileset.get_pixel(10*ts,ts161);
                color[TITLE_DISABLED_BR] = tileset.get_pixel(11*ts+ts1,ts161);
            }
        }

    void draw_tile(Surface &surf, int tx, int ty, SDL_Point dst) {
        tileset.blit(surf,Rect(tilesize*tx,tilesize*ty,tilesize,tilesize),
                          Rect(dst.x,dst.y,tilesize,tilesize));
    }
    void draw_tiles(Surface &surf, int tx, int ty, int tw, int th, SDL_Point dst) {
        tileset.blit(surf,Rect(tilesize*tx,tilesize*ty,tilesize*tw,tilesize*th),
                          Rect(dst.x,dst.y,tilesize*tw,tilesize*th));
    }
    void draw_repeat(Surface &surf, int tx, int ty, SDL_Rect dst) {
        int dx,dy;
        int x2=dst.x+dst.w;
        int y2=dst.y+dst.h;
        for (dx=dst.x; dx+tilesize<=x2; dx+=tilesize) {
            for (dy=dst.y; dy+tilesize<=y2; dy+=tilesize) {
                tileset.blit(surf,Rect(tilesize*tx,tilesize*ty,tilesize,tilesize),
                                  Rect(dx,dy,tilesize,tilesize));
            }
            if (y2-dy>0) {
                tileset.blit(surf,Rect(tilesize*tx,tilesize*ty,tilesize,y2-dy),
                                  Rect(dx,dy,tilesize,y2-dy));
            }
        }
        if (x2-dx>0) {
            for (dy=dst.y; dy+tilesize<=y2; dy+=tilesize) {
                tileset.blit(surf,Rect(tilesize*tx,tilesize*ty,x2-dx,tilesize),
                                  Rect(dx,dy,x2-dx,tilesize));
            }
            if (y2-dy>0) {
                tileset.blit(surf,Rect(tilesize*tx,tilesize*ty,x2-dx,y2-dy),
                                  Rect(dx,dy,x2-dx,y2-dy));
            }
        }
    }
    void draw_hbar(Surface &surf, int left, int ty, SDL_Rect dst) { // Uses 3x1 tiles!
        if (dst.w<=2*tilesize) {
            int mid=dst.w/2;
            tileset.blit(surf,Rect(tilesize*left,tilesize*ty,mid,tilesize),
                              Rect(dst.x,dst.y,mid,tilesize));
            tileset.blit(surf,Rect(tilesize*(left+3)-(dst.w-mid),tilesize*ty,dst.w-mid,tilesize),
                              Rect(dst.x+mid,dst.y,dst.w-mid,tilesize));
        } else {
            draw_repeat(surf,left+1,ty,Rect(dst.x+tilesize,dst.y,dst.w-2*tilesize,tilesize));
            draw_tile(surf,left+2,ty,Point(dst.x+dst.w-tilesize,dst.y));
            draw_tile(surf,left,ty,Point(dst.x,dst.y));
        }
    }
    void draw_vbar(Surface &surf, int tx, int top, SDL_Rect dst) { // Uses 1x3 tiles!
        if (dst.h<=2*tilesize) {
            int mid=dst.h/2;
            tileset.blit(surf,Rect(tilesize*tx,tilesize*top,tilesize,mid),
                              Rect(dst.x,dst.y,tilesize,mid));
            tileset.blit(surf,Rect(tilesize*tx,tilesize*(top+3)-(dst.h-mid),tilesize,dst.h-mid),
                              Rect(dst.x,dst.y+mid,tilesize,dst.h-mid));
        } else {
            draw_repeat(surf,tx,top+1,Rect(dst.x,dst.y+tilesize,tilesize,dst.h-2*tilesize));
            draw_tile(surf,tx,top+2,Point(dst.x,dst.y+dst.h-tilesize));
            draw_tile(surf,tx,top,Point(dst.x,dst.y));
        }
    }
    void draw_frame(Surface &surf, int left, int top, SDL_Rect dst) { // Uses 3x3 tiles!
        // Get border size
        int x1,x2,y1,y2;
        x1=tilesize;
        x2=dst.w-tilesize;
        y1=tilesize;
        y2=dst.h-tilesize;
        if (x1>x2) x1=x2=(x1+x2)/2;
        if (y1>y2) y1=y2=(y1+y2)/2;
        int w2=dst.w-x2;
        int h2=dst.h-y2;
        // Corners
        tileset.blit(surf,Rect(tilesize*left,tilesize*top,x1,y1),
                          Rect(dst.x,dst.y,x1,y1));
        tileset.blit(surf,Rect(tilesize*(left+3)-w2,tilesize*top,w2,y1),
                          Rect(dst.x+x2,dst.y,w2,y1));
        tileset.blit(surf,Rect(tilesize*left,tilesize*(top+3)-h2,x1,h2),
                          Rect(dst.x,dst.y+y2,x1,h2));
        tileset.blit(surf,Rect(tilesize*(left+3)-w2,tilesize*(top+3)-h2,w2,h2),
                          Rect(dst.x+x2,dst.y+y2,w2,h2));
        // Horizontal Borders
        int dx,dy; // Don't inline! Will be used after the loops for fractional tiles
        for (dx=x1; dx+tilesize<=x2; dx+=tilesize) {
            tileset.blit(surf,Rect(tilesize*(left+1),tilesize*top,tilesize,y1),
                              Rect(dst.x+dx,dst.y,tilesize,y1));
            tileset.blit(surf,Rect(tilesize*(left+1),tilesize*(top+3)-h2,tilesize,h2),
                              Rect(dst.x+dx,dst.y+y2,tilesize,h2));
        }
        if (x2-dx>0) {
            tileset.blit(surf,Rect(tilesize*(left+1),tilesize*top,x2-dx,y1),
                              Rect(dst.x+dx,dst.y,x2-dx,y1));
            tileset.blit(surf,Rect(tilesize*(left+1),tilesize*(top+3)-h2,x2-dx,h2),
                              Rect(dst.x+dx,dst.y+y2,x2-dx,h2));
        }
        // Vertical Borders
        for (dy=y1; dy+tilesize<=y2; dy+=tilesize) {
            tileset.blit(surf,Rect(tilesize*left,tilesize*(top+1),x1,tilesize),
                              Rect(dst.x,dst.y+dy,x1,tilesize));
            tileset.blit(surf,Rect(tilesize*(left+3)-w2,tilesize*(top+1),w2,tilesize),
                              Rect(dst.x+x2,dst.y+dy,w2,tilesize));
        }
        if (y2-dy>0) {
            tileset.blit(surf,Rect(tilesize*left,tilesize*(top+1),x1,y2-dy),
                              Rect(dst.x,dst.y+dy,x1,y2-dy));
            tileset.blit(surf,Rect(tilesize*(left+3)-w2,tilesize*(top+1),w2,y2-dy),
                              Rect(dst.x+x2,dst.y+dy,w2,y2-dy));
        }
        // Center Area
        draw_repeat(surf,left+1,top+1,Rect(dst.x+x1,dst.y+y1,x2-x1,y2-y1));
    }
    void draw_frame(Surface &surf, int left, int top) {draw_frame(surf,left,top,surf.rect());}

    void draw_titlebar(Surface &surf, SDL_Rect dst, bool enabled = true) {
        const int WIDTH = 256;
        if (enabled && !titlebar_enabled) {
            int h = tilesize+bordersize;
            titlebar_enabled = Surface(WIDTH,h);
            for (int y = 0; y < h; y++)
                for (int x = 0; x < WIDTH; x++) {
                    float px = (float)x/WIDTH;
                    float py = (float)y/h;
                    titlebar_enabled.set_pixel(x,y,mix(mix(color[TITLE_ENABLED_BR],color[TITLE_ENABLED_BL],px),
                                                       mix(color[TITLE_ENABLED_TR],color[TITLE_ENABLED_TL],px),py));
            }
        } else if (!enabled && !titlebar_disabled) {
            int h = tilesize+bordersize;
            titlebar_disabled = Surface(WIDTH,h);
            for (int y = 0; y < h; y++)
                for (int x = 0; x < WIDTH; x++) {
                    float px = (float)x/WIDTH;
                    float py = (float)y/h;
                    titlebar_disabled.set_pixel(x,y,mix(mix(color[TITLE_DISABLED_BR],color[TITLE_DISABLED_BL],px),
                                                       mix(color[TITLE_DISABLED_TR],color[TITLE_DISABLED_TL],px),py));
            }
        }
        if (enabled) titlebar_enabled.blit_scaled(surf,titlebar_enabled.rect(),dst);
        else titlebar_disabled.blit_scaled(surf,titlebar_disabled.rect(),dst);
    }

    void draw_titlebar(Surface &surf, bool enabled = true) {
        draw_titlebar(surf,Rect(bordersize,bordersize,surf.width()-2*bordersize, tilesize+bordersize),enabled);}

    Surface tileset;
    Font font;
    int tilesize,bordersize;
    int cursor_blink = 666; // 4/3s feels about right
    Surface titlebar_enabled, titlebar_disabled;

    enum COLORS : unsigned {
        WINDOW_BACKGROUND = 0, EDIT_BACKGROUND, TEXT_ENABLED, TEXT_DISABLED,
        TEXT_SELECTED, TEXT_SELECTION, TITLE_ENABLED_TEXT, TITLE_DISABLED_TEXT,
        TITLE_ENABLED_TL, TITLE_ENABLED_TR, TITLE_ENABLED_BL, TITLE_ENABLED_BR,
        TITLE_DISABLED_TL, TITLE_DISABLED_TR, TITLE_DISABLED_BL, TITLE_DISABLED_BR,
        COLORS_MAX};
    // Default is Windows 95 Theme
    SDL_Color color[COLORS_MAX] = { {192,192,192,255}, {255,255,255,255}, {0,0,0,255},       {128,128,128,255},
                                    {255,255,255,255}, {0,0,128,255},     {255,255,255,255}, {192,192,192,255},
                                    {0,0,128,255},     {0,0,128,255},     {0,0,128,255},     {0,0,128,255},
                                    {128,128,128,255}, {128,128,128,255}, {128,128,128,255}, {128,128,128,255} };

};

class Widget {
public:
    Widget(Widget *_parent, SDL_Rect _area) : parent(_parent), area(_area)
        {if (parent) parent->child.push_back(this);}
    virtual ~Widget() {};

    virtual Theme &get_theme() throw(Error) {
        if ((bool)custom_theme) return *custom_theme;
        if (parent) return parent->get_theme();
        else throw Error("qdw::Widget has no theme.");}
    virtual Theme &get_tileset() throw(Error) {
        if ((bool)custom_theme && custom_theme->tilesize>0) return *custom_theme;
        if (parent) return parent->get_tileset();
        else throw Error("qdw::Widget has no tileset.");}
    virtual Gui &get_gui() throw(Error) {
        if (parent) return parent->get_gui();
        else throw Error("qdw::Widget is not assigned to a Gui.");
        }

    virtual bool hit(const SDL_Event &e) {
        bool consumed = false;
        if (!enabled()) return Inside(e,area);
        if (Inside(e,area)) consumed = mouse_event_hit(e,Point(area.x,area.y));
        else miss(e);
        switch (e.type) {
        case SDL_KEYDOWN:
            if (keyDown && (e.key.keysym.sym == hotkey.sym) && my_keymods(e.key.keysym.mod)) {
                    keyDown(this,e);
                    consumed = true;
            }
            for (Widget *w : child) consumed|=w->hit(e);
            break;
        case SDL_KEYUP:
            if (keyUp && (e.key.keysym.sym == hotkey.sym) && my_keymods(e.key.keysym.mod)) {
                    keyUp(this,e);
                    consumed = true;
            }
            for (Widget *w : child) consumed|=w->hit(e);
            break;
        }
        return consumed;
    };

    virtual void miss(const SDL_Event &e) { // Inform widgets about events that can't hit them (but may reset state)
        switch (e.type) {
        case SDL_MOUSEBUTTONDOWN:
            for (Widget *w : child) w->miss(e);
            break; // Will never do anything
        case SDL_MOUSEBUTTONUP:
            for (Widget *w : child) w->miss(e);
            release();
            break;
        case SDL_MOUSEMOTION:
            for (Widget *w : child) w->miss(e);
            if (selected()) {
                unselect();
                if (pressed()) release();
                if (mouseLeave) mouseLeave(this,e-Point(area.x,area.y));
            }
            break;
        case SDL_KEYDOWN:
            break;
        case SDL_KEYUP:
            for (Widget *w : child) w->miss(e);
            break;
        }
    }

    virtual bool inject(const SDL_Event &e) {return false;} // Targets this element specifically; will not get passed down to children.

    virtual bool update() {
        if (!visible()) {
            if (state&ST_REDRAW) {
                state&=~ST_REDRAW;
                return true;
            } else return false;
        }
        pre_update();
        for (Widget *w : child) if (w->update()) state|=ST_REDRAW;
        if (state&ST_REDRAW) {
            surface = self();
            for (Widget *w : child) w->draw(surface);
            state&=~ST_REDRAW;
            return true;
        } else return false;
    }

    bool mouse_event_hit(const SDL_Event &e, SDL_Point offset) {
        bool consumed = false;
        for (Widget *w : reverse(child)) if (consumed) w->miss(e-offset); else consumed |= w->hit(e-offset);
        switch (e.type) {
        case SDL_MOUSEBUTTONDOWN:
            if (!consumed) {
                if (e.button.button == SDL_BUTTON_LEFT) {
                    press();
                    consumed = true;
                    if (mouseDblClick && (e.button.clicks == 2)) {
                        mouseDblClick(this,e-offset);
                    }
                }
                if (mouseDown) {
                    mouseDown(this,e-offset);
                    consumed = true;
                }
            }
            break;
        case SDL_MOUSEBUTTONUP:
            if (!consumed) {
                if (mouseUp) {
                    mouseUp(this,e-offset);
                    consumed = true;
                }
                if (mouseClick && pressed() && (e.button.button == SDL_BUTTON_LEFT)) {
                    mouseClick(this,e-offset);
                    consumed = true;
                }
                if (current_focus() && current_focus()->dragging() && (e.button.button == SDL_BUTTON_LEFT)) {
                    Widget *f=current_focus();
                    if (f!=this) { // Can't drop something on itself.
                        if (f->mouseDragStop) f->mouseDragStop(this,e-offset);
                        if (f->mouseDrop) {
                            f->mouseDrop(f,this,e-offset);
                            consumed = true;
                        }
                        handle_drop(f);
                        if (mouseDrop) {
                            mouseDrop(f,this,e-offset);
                            consumed = true;
                        }
                        f->state&=~ST_DRAGGING;
                    }
                }
            }
            release();
            break;
        case SDL_MOUSEMOTION:
            if (!consumed) {
                if (!selected()) select();
                if (selected() && mouseMove) {
                    mouseMove(this,e-offset);
                }
                if (!selected() && mouseEnter) {
                    mouseEnter(this,e-offset);
                }
                consumed = true;
            }
            break;
        }
        return consumed;
    }

    virtual Surface self() = 0;

    virtual void draw(Surface target) {
        if (state&ST_VISIBLE) {
            SDL_Rect src = area;
            src.x = src.y = 0;
            surface.blit(target,src,area);
        }
    }

    virtual SDL_Point abs_pos() {
        SDL_Point tl = {0,0};
        if (parent) tl = parent->abs_pos();
        tl.x+=area.x;
        tl.y+=area.y;
        return tl;
    }

    virtual void handle_drop(Widget *dropped) {};

    bool visible() {
        if (parent&& !parent->visible()) return false;
        else return state&ST_VISIBLE;
    }
    bool enabled() {
        if (parent&& !parent->enabled()) return false;
        else return state&ST_ENABLED;
    }
    bool pressed() {return state&ST_CLICKED;}
    bool selected() {return state&ST_SELECTED;}
    bool checked() {return state&ST_CHECKED;}
    bool highlighted() {return state&ST_HIGHLIGHTED;}
    bool dragging() {return state&ST_DRAGGING;}
    bool resizing() {return state&ST_RESIZING;}

    // Redraw can be expensive, so only set it if state changes...
    void show() {if (!(state&ST_VISIBLE)) state|=(ST_VISIBLE|ST_REDRAW);}
    void hide() {if (state&ST_VISIBLE) {state&=~ST_VISIBLE; state|=ST_REDRAW;}}
    void enable() {if (!(state&ST_ENABLED)) {state|=(ST_ENABLED); request_redraw(true);}}
    void disable() {if (state&ST_ENABLED) {state&=~ST_ENABLED; request_redraw(true);}}
    void press() {if (!(state&ST_CLICKED)) state|=(ST_CLICKED|ST_REDRAW);}
    void release() {if (state&ST_CLICKED) {state&=~ST_CLICKED; state|=ST_REDRAW;}}
    void select() {if (!(state&ST_SELECTED)) state|=(ST_SELECTED|ST_REDRAW);}
    void unselect() {if (state&ST_SELECTED) {state&=~ST_SELECTED; state|=ST_REDRAW;}}
    void check() {if (!(state&ST_CHECKED)) state|=(ST_CHECKED|ST_REDRAW);}
    void uncheck() {if (state&ST_CHECKED) {state&=~ST_CHECKED; state|=ST_REDRAW;}}
    void highlight() {if (!(state&ST_HIGHLIGHTED)) state|=(ST_HIGHLIGHTED|ST_REDRAW);}
    void unhlight() {if (state&ST_HIGHLIGHTED) {state&=~ST_HIGHLIGHTED; state|=ST_REDRAW;}}

    virtual void close(bool recursive = true) {
        state|=ST_CLOSE;
        if (recursive) for (Widget *w : child) w->close(true);}
    virtual void request_redraw(bool recursive = false) {
        state|=ST_REDRAW;
        if (recursive) for (Widget *w : child) w->request_redraw(true);}
    virtual void resize_update() {return;}

    bool remove_child(Widget *ch) {
        for (auto wit = child.begin(); wit!=child.end(); wit++) { // I need the iterator here!
            if (*wit==ch) {
                child.erase(wit);
                ch->parent = nullptr;
                request_redraw();
                return true;
            }
        }
        return false;
    }
    bool add_child(Widget *ch) {
        if (ch->parent && ch->parent!=this) return false;
        for (Widget *w : child) {
            if (w==ch) return false;
        }
        child.push_back(ch);
        ch->parent = this;
        request_redraw();
        return true;
    }
    bool change_gui(Gui &ng);
    bool change_parent(Widget *np) {
        if (np==parent) return true;
        SDL_Point opos = abs_pos();
        if (&get_gui()!=&(np->get_gui()) &&
            !change_gui(np->get_gui())) return false;
        parent->remove_child(this);
        return np->add_child(this);
        SDL_Point npos = abs_pos();
        area.x+=opos.x-npos.x;
        area.y+=opos.y-npos.y;
    }

    virtual Widget *current_focus() {return parent?parent->current_focus():nullptr;}
    virtual void take_focus(); // Need to be defined after Gui
    virtual void lose_focus();
    bool has_focus();

    bool on_top() {return parent && (parent->child.back())==this;}
    void to_top() {if (Widget *p=parent) {p->remove_child(this); p->add_child(this); p->request_redraw(true);} return;}

    virtual void pre_update() {
        if (hotkey.mod && my_keymods(SDL_GetModState())) highlight();
        else unhlight(); // Need to do this last minute, because mod keys don't generate SDL_KEYDOWN events
    }

    virtual SDL_Rect move_limits() {return area;}

    std::string get_id();

    void setHotkey(SDL_Keycode key, Uint16 mod, Callback onDown, Callback onUp = nullptr) {
    hotkey.sym = key; hotkey.mod = mod; keyDown = onDown; keyUp = onUp;}

    void addFrame(std::string id, SDL_Rect area, int lvl) throw(Error);
    void addHScroll(std::string id, SDL_Rect area) throw(Error);
    void addVScroll(std::string id, SDL_Rect area) throw(Error);
    void addScrollFrame(std::string id, SDL_Rect area, int lvl) throw(Error);
    void addLabel(std::string id, SDL_Rect area, const std::string &caption) throw(Error);
    void addButton(std::string id, SDL_Rect area, const std::string &caption, Callback onClick) throw(Error);
    void addSimpleButton(std::string id, SDL_Rect area, int icon_x, int icon_y, bool framed, Callback onClick) throw(Error);
    void addCheckBox(std::string id, SDL_Rect area, const std::string &caption) throw(Error);
    void addOptionButton(std::string id, SDL_Rect area, const std::string &caption, int group) throw(Error);
    void addEditBox(std::string id, SDL_Rect area) throw(Error);
    void addIcon(std::string id, SDL_Rect area, Surface surf, const std::string &_caption, Callback onDblClick) throw(Error);
    void addSubWindow(std::string id, SDL_Rect area, const std::string &title) throw(Error);
    void addTextBox(std::string id, SDL_Rect area) throw(Error);
    void addGrid(std::string id, SDL_Rect area, int cellw, int cellh, int lvl) throw(Error);

    Widget *parent = nullptr;
    std::vector<Widget*> child;

    unsigned state = ST_VISIBLE|ST_ENABLED|ST_REDRAW;
    enum STATES : unsigned { // Bitfield
        ST_VISIBLE = 0x1,
        ST_ENABLED = 0x2,
        ST_CLICKED = 0x4,
        ST_SELECTED = 0x8,
        ST_CHECKED = 0x10,
        ST_HIGHLIGHTED = 0x20,
        ST_CLOSE = 0x40,
        ST_REDRAW = 0x80,
        ST_DRAGGABLE = 0x100,
        ST_DRAGGING = 0x200,
        ST_RESIZABLE = 0x400,
        ST_RESIZING = 0x800,
        ST_MINABLE = 0x1000,
        ST_MAXABLE = 0x2000
    };

    unsigned type = TP_NONE;
    enum TYPES : unsigned {TP_NONE = 0, TP_GUI, TP_FRAME, TP_LABEL, TP_BUTTON, TP_SBUTTON, TP_ICON, TP_GRID, TP_HSCROLL, TP_VSCROLL,
                           TP_SCROLLFRAME, TP_CHECKBOX, TP_OPTIONBUTTON, TP_SUBWINDOW, TP_EDITBOX, TP_TEXTBOX};

    std::shared_ptr<Theme> custom_theme;
    SDL_Rect area;
    float h_align = .5f;
    float v_align = .5f;
    SDL_Keysym hotkey;
    bool strict_hotkey_mod = false;
    Surface surface;
    int minw = 0;
    int minh = 0;

    // Primitive Mouse Callbacks
    Callback mouseDown = nullptr;
    Callback mouseUp = nullptr;
    Callback mouseMove = nullptr;
    Callback mouseScroll = nullptr;

    // Complex Mouse Callbacks
    Callback mouseClick = nullptr;
    Callback mouseDblClick = nullptr;
    Callback mouseEnter = nullptr;
    Callback mouseLeave = nullptr;
    Callback mouseResize = nullptr;
    Callback mouseResizeStart = nullptr;
    Callback mouseResizeStop = nullptr;
    Callback mouseDrag = nullptr;
    Callback mouseDragStart = nullptr;
    Callback mouseDragStop = nullptr;
    Interaction mouseDrop = nullptr;

    // Primitive Key Callbacks
    Callback keyDown = nullptr;
    Callback keyUp = nullptr;

    // Complex Key Callbacks
    Callback textEdit = nullptr;

    std::string user_data;

    bool my_keymods(Uint8 km) {
        return (km == hotkey.mod) || (!strict_hotkey_mod && similar(km,hotkey.mod));
    }
};

class Gui : public Widget {
public:
    Gui(Theme _theme, SDL_Rect _area) throw(Error) :
        Widget(nullptr,_area), theme(_theme) {type = TP_GUI; if (!theme.tilesize) throw Error("Gui needs a valid tileset.");}

    ~Gui() {for (auto w : owned_widgets) delete w.second;}; // Only delete what you own...

    virtual Theme &get_theme() throw() {return theme;}
    virtual Theme &get_tileset() throw() {return theme;}
    virtual Gui &get_gui() throw() {return *this;}

    virtual Widget *current_focus() {return focus;}

    virtual bool hit(const SDL_Event &e) {
        if (!enabled()) return false;
        bool consumed = false;
        if (focus) {
            if ((e.type == SDL_MOUSEMOTION) &&  (e.motion.state&SDL_BUTTON_LMASK)) {
                if (focus->dragging()) {
                    SDL_Rect limits = focus->parent->move_limits();
                    focus->area.x+=e.motion.xrel;
                    focus->area.y+=e.motion.yrel;
                    if (focus->area.x+focus->area.w>limits.w)
                        focus->area.x = limits.w-focus->area.w;
                    if (focus->area.y+focus->area.h>limits.h)
                        focus->area.y = limits.h-focus->area.h;
                    if (focus->area.x<limits.x) focus->area.x = limits.x;
                    if (focus->area.y<limits.y) focus->area.y = limits.y;
                    if (focus->mouseDrag) focus->mouseDrag(focus,e);
                    focus->request_redraw();
                    consumed = true;
                } else if (focus->resizing()) {
                    int &tsz = focus->get_tileset().tilesize;
                    int &bsz = focus->get_tileset().bordersize;
                    int minsz = tsz+2*bsz;
                    SDL_Rect limits = focus->parent->move_limits();
                    focus->area.w+=e.motion.xrel;
                    focus->area.h+=e.motion.yrel;
                    if (focus->area.w<focus->minw+minsz) focus->area.w = focus->minw+minsz;
                    if (focus->area.h<focus->minh+minsz) focus->area.h = focus->minh+minsz;
                    if (focus->area.x+focus->area.w>limits.w)
                        focus->area.w = limits.w-focus->area.x;
                    if (focus->area.y+focus->area.h>limits.h)
                        focus->area.h = limits.h-focus->area.y;
                    focus->resize_update();
                    if (focus->mouseResize) focus->mouseResize(focus,e);
                    focus->request_redraw();
                    consumed = true;
                }
            }
            if (!consumed) consumed = focus->inject(e);
        }
        if (consumed) Widget::miss(e);
        else consumed |= Widget::hit(e);
        for (auto &p : reverse(owned_widgets)) {
            if (p.second->state&ST_CLOSE) {
                if (focus == p.second) focus = nullptr;
                for (Widget *c : p.second->child) {
                    if (c->state&ST_CLOSE) c->parent = nullptr;
                    else {
                        c->change_parent(p.second->parent);
                        if (!c->parent) c->close();
                    }
                }
                if (p.second->parent) p.second->parent->remove_child(p.second);
                delete p.second;
                owned_widgets.erase(p.first);
            }
        }
        return consumed;
    }

    virtual bool update() {
        if (focus && !focus->enabled()) focus->lose_focus();
        return Widget::update();
    }

    virtual Surface self() {
        Surface surf(area.w,area.h);
        return surf;
    }

    void add(std::string id, Widget *new_member) throw(Error) {
        if (owned_widgets.count(id)) throw Error("qdw::Widget with this ID already exists in this Gui.");
        if (new_member->type==TP_GUI) throw Error("qdw::Gui can't own another Gui.");
        owned_widgets[id]=new_member;
    }
    // Do not use during hit/update/draw! Use Widget::close() instead!
    bool del(std::string id, bool recursive = true) {
        if (owned_widgets.count(id)) {
            Widget *todel = owned_widgets[id];
            if (recursive) for (Widget *w : todel->child) del(w,true);
            if (focus == todel) focus = nullptr;
            todel->parent->remove_child(todel);
            delete todel;
            owned_widgets.erase(id);
            return true;
        } else return false;
    }
    bool del(Widget *member, bool recursive = true) {
        for (auto p : owned_widgets) {
            if (p.second == member) {
                if (recursive) for (Widget *w : member->child) del(w,true);
                if (focus == member) focus = nullptr;
                member->parent->remove_child(member);
                delete member;
                owned_widgets.erase(p.first);
                return true;
            }
        }
        return false;
    }
    template <typename T = Widget>
    T* get(std::string id) {
        if (owned_widgets.count(id)) return static_cast<T*>(owned_widgets[id]);
        else return nullptr;
    }
    Widget& operator[](std::string id) {return *get(id);}
    std::string get_id(Widget *member) {
        if (member==this) return "GUI";
        for (auto p : owned_widgets)
            if (p.second == member) return p.first;
        return "";
    }



    Theme theme;
    Widget *focus = nullptr;

    std::map<std::string,Widget*> owned_widgets;
};

inline void Widget::take_focus() {Widget *f = get_gui().focus; if (f) f->lose_focus(); get_gui().focus = this;}
inline void Widget::lose_focus() {if (this==get_gui().focus) {get_gui().focus = nullptr; request_redraw();}}
inline bool Widget::has_focus() {return (this == get_gui().focus);}

inline std::string Widget::get_id() {return get_gui().get_id(this);}

inline bool Widget::change_gui(Gui &ng) {
    Gui &og = get_gui();
    std::string id = og.get_id(this);
    if (ng.owned_widgets.count(id)) return false;
    og.owned_widgets.erase(id);
    ng.owned_widgets[id]=this;
    return true;
};

class TexGui : public Gui {
public:
    TexGui(Renderer _renderer, Theme _theme, SDL_Rect _area) :
        Gui(_theme,_area), renderer(_renderer) {}

    virtual bool update() {
        if (Gui::update()) {
            texture = Texture(renderer,surface);
            return true;
        } else return false;
    }

    void render() {
        if (state&ST_VISIBLE) {
            SDL_Rect src = area;
            src.x = src.y = 0;
            texture.render(src,area);
        }
    }

    Renderer renderer;
    Texture texture;
};

class Frame : public Widget {
public:
    Frame(Widget *_parent, SDL_Rect _area, int lvl = 0) :
        Widget(_parent,_area), level(lvl) {type = TP_FRAME;}

    virtual Surface self() {
        Surface surf(area.w,area.h);
        if (level>=-2 && level <=2)
            get_tileset().draw_frame(surf,6+3*level,0);
        return surf;
    }

    virtual SDL_Rect move_limits() {
        SDL_Rect temp = area;
        int bsz = get_tileset().bordersize;
        temp.x=bsz;
        temp.y=bsz;
        temp.w-=bsz;
        temp.h-=bsz;
        return temp;
    }

    int level;
};
inline void Widget::addFrame(std::string id, SDL_Rect area, int lvl = 0) throw(Error) {
        get_gui().add(id, new Frame(this,area,lvl));}

class HScroll : public Widget {
public:
    HScroll(Widget *_parent, SDL_Rect _area) :
        Widget(_parent,_area) {type = TP_HSCROLL;}

    virtual Surface self() {
        Surface surf(area.w,area.h);
        float p=position;
        if (p<0) p=0.f;
        if (p>1) p=1.f;
        p=p/(1.f+slider);
        Theme &t = get_tileset();
        int yoff=3;
        if (!enabled()) yoff=6;
        else if (pressed()) yoff=9;
        int scrollw = area.w-2*t.tilesize;
        t.draw_hbar(surf,9,yoff,Rect(0,(area.h-t.tilesize)*v_align,area.w,t.tilesize));
        if (scrollw>0) t.draw_hbar(surf,9,yoff+1,Rect(t.tilesize+scrollw*p,(area.h-t.tilesize)*v_align,scrollw*slider/(1.f+slider)+1,t.tilesize));
        return surf;
    }

    virtual bool hit(const SDL_Event &e) {
        if (!enabled()) return Inside(e,area);
        bool consumed = false;
        if ((e.type == SDL_MOUSEMOTION) && pressed() && Inside(Point(e.motion.x,e.motion.y),area)) {
            int tsz = get_tileset().tilesize;
            if (e.motion.x<=area.x+tsz) position=0.f;
            else if (e.motion.x+tsz>=area.x+area.w) position=1.f;
            else position=(e.motion.x-area.x-tsz)/(float)(area.w-2*tsz);
            state|=ST_REDRAW;
            consumed = true;
        }
        if ((e.type == SDL_MOUSEBUTTONDOWN) && Inside(Point(e.button.x,e.button.y),area)) {
            int tsz = get_tileset().tilesize;
            if (e.button.x<=area.x+tsz) position=0.f;
            else if (e.button.x+tsz>=area.x+area.w) position=1.f;
            else position=(e.button.x-area.x-tsz)/(float)(area.w-2*tsz);
            state|=ST_REDRAW;
            consumed = true;
        }
        consumed |= Widget::hit(e); // Default stuff
        return consumed;
    }

    float position = 0.f;
    float slider = 0.125f;
};
inline void Widget::addHScroll(std::string id, SDL_Rect area) throw(Error) {
        get_gui().add(id, new HScroll(this,area));}

class VScroll : public Widget {
public:
    VScroll(Widget *_parent, SDL_Rect _area) :
        Widget(_parent,_area) {type = TP_HSCROLL;}

    virtual Surface self() {
        Surface surf(area.w,area.h);
        float p=position;
        if (p<0) p=0.f;
        if (p>1) p=1.f;
        p=p/(1.f+slider);
        Theme &t = get_tileset();
        int yoff=3;
        if (!enabled()) yoff=6;
        else if (pressed()) yoff=9;
        int scrollh = area.h-2*t.tilesize;
        t.draw_vbar(surf,7,yoff,Rect((area.w-t.tilesize)*h_align,0,t.tilesize,area.h));
        if (scrollh>0) t.draw_vbar(surf,8,yoff,Rect((area.w-t.tilesize)*h_align,t.tilesize+scrollh*p,t.tilesize,scrollh*slider/(1.f+slider)+1));
        return surf;
    }

    virtual bool hit(const SDL_Event &e) {
        if (!enabled()) return Inside(e,area);
        bool consumed = false;
        if ((e.type == SDL_MOUSEMOTION) && pressed() && Inside(Point(e.motion.x,e.motion.y),area)) {
            int tsz = get_tileset().tilesize;
            if (e.motion.y<=area.y+tsz) position=0.f;
            else if (e.motion.y+tsz>=area.y+area.h) position=1.f;
            else position=(e.motion.y-area.y-tsz)/(float)(area.h-2*tsz);
            state|=ST_REDRAW;
            consumed = true;
        }
        if ((e.type == SDL_MOUSEBUTTONDOWN) && Inside(Point(e.button.x,e.button.y),area)) {
            int tsz = get_tileset().tilesize;
            if (e.button.y<=area.y+tsz) position=0.f;
            else if (e.button.y+tsz>=area.y+area.h) position=1.f;
            else position=(e.button.y-area.y-tsz)/(float)(area.h-2*tsz);
            state|=ST_REDRAW;
            consumed = true;
        }
        consumed |= Widget::hit(e); // Default stuff
        return consumed;
    }

    float position = 0.f;
    float slider = 0.125f;
};
inline void Widget::addVScroll(std::string id, SDL_Rect area) throw(Error) {
        get_gui().add(id, new VScroll(this,area));}

class ScrollFrame : public Frame {
public:
    ScrollFrame(Widget *_parent, SDL_Rect _area, int lvl = -2) :
        Frame(_parent,_area,lvl), scrollx(nullptr, _area), scrolly(nullptr, _area)
            {type = TP_SCROLLFRAME; scrollx.parent=scrolly.parent=this; h_align=v_align=0.f;
             shown_area=inner_area=area; minw=minh=2*get_tileset().tilesize;}

    virtual bool hit(const SDL_Event &e) {
        bool consumed = false;
        if (!enabled()) return Inside(e,area);
        SDL_Point tl = Point(area.x,area.y);
        consumed |= scrollx.hit(e-tl);
        consumed |= scrolly.hit(e-tl);
        if (consumed) {
            int wrange = std::max(0,inner_area.w-shown_area.w);
            int hrange = std::max(0,inner_area.h-shown_area.h);
            xoff=wrange*scrollx.position;
            yoff=hrange*scrolly.position;
            fix_offsets();
        }

        if (e.type == SDL_MOUSEBUTTONDOWN) {
            if (rbtn() || sbtn()) {
                Theme &tiles = get_tileset();
                int &bsz = tiles.bordersize;
                int &tsz = tiles.tilesize;
                if (Inside(e,Rect(area.x+area.w-bsz-tsz,area.y+area.h-bsz-tsz,tsz,tsz))) {
                    if (rbtn() && (state&ST_RESIZABLE)) {
                        state|=ST_RESIZING;
                        if (mouseResizeStart) mouseResizeStart(this,e);
                        take_focus();
                        consumed = true;
                    } else if (sbtn() && (scrollx.enabled() || scrolly.enabled())) {
                        take_focus();
                        consumed = true;
                    }
                }
            }
        }

        if (e.type==SDL_KEYDOWN || e.type==SDL_KEYDOWN) consumed |= Widget::hit(e);
        if (Inside(e,shown_area+tl)) {
            consumed |= mouse_event_hit(e,Point(area.x+shown_area.x-xoff,area.y+shown_area.y-yoff));
        } else Widget::miss(e);
        return consumed;
    }

    virtual bool inject(const SDL_Event &e) {
        if (!enabled() || (e.type == SDL_MOUSEBUTTONUP)) {
            lose_focus();
            if (resizing()) {
                state&=~ST_RESIZING;
                if (mouseResizeStop) mouseResizeStop(this,e);
            }
            if (dragging()) {
                state&=~ST_DRAGGING;
                if (mouseDragStop) mouseDragStop(this,e);
            }
            request_redraw();
            return false;
        }
        if ((e.type == SDL_MOUSEMOTION) &&  (e.motion.state&SDL_BUTTON_LMASK)) {
            if (!(state&(ST_DRAGGING|ST_RESIZING))) {
                xoff+=e.motion.xrel;
                yoff+=e.motion.yrel;
                fix_offsets();
                return true;
            }
        }
        return false;
    }

    virtual void resize_update() {
        scrollx.position=std::min(1.f,xoff/((float)inner_area.w-shown_area.w));
        scrolly.position=std::min(1.f,yoff/((float)inner_area.h-shown_area.h));
        scrollx.slider=std::min(1.f,shown_area.w/(float)inner_area.w);
        scrolly.slider=std::min(1.f,shown_area.h/(float)inner_area.h);
        scrollx.request_redraw();
        scrolly.request_redraw();
        return;
    }

    virtual bool update() {
        if (!visible()) {
            if (state&ST_REDRAW) {
                state&=~ST_REDRAW;
                return true;
            } else return false;
        }
        for (Widget *w : child) if (w->update()) state|=ST_REDRAW;
        if ((scrollx.state&ST_REDRAW) || (scrolly.state&ST_REDRAW)) state|=ST_REDRAW;
        if (state&ST_REDRAW) {

            surface = self();

            update_areas();
            resize_update();

            draw_scrollbars();

            state&=~ST_REDRAW;
            return true;
        } else return false;
    }

    virtual void draw_scrollbars() {
        Theme &tiles=get_tileset();
        int tsz = tiles.tilesize;
        int bsz = tiles.bordersize;

        scrollx.area=Rect(shown_area.x,shown_area.y+shown_area.h,shown_area.w,tsz);
        scrolly.area=Rect(shown_area.x+shown_area.w,shown_area.y,tsz,shown_area.h);
        if (rbtn() || sbtn()) {
            if (!scrollx.visible()) scrolly.area.h-=tsz;
            if (!scrolly.visible()) scrollx.area.w-=tsz;
        }
        scrollx.update(); scrolly.update();

        Surface inner_surface(inner_area.w,inner_area.h);
        for (Widget *w : child) w->draw(inner_surface);
        inner_surface.blit(surface,Rect(xoff,yoff,shown_area.w,shown_area.h),shown_area);

        if (scrollx.visible()) scrollx.draw(surface);
        if (scrolly.visible()) scrolly.draw(surface);

        if (scrollx.visible()&&scrolly.visible())
            surface.fill_rect(Rect(area.w-bsz-tsz,area.h-bsz-tsz,tsz,tsz),tiles.color[Theme::WINDOW_BACKGROUND]);

        if (rbtn()) {
            if (state&ST_RESIZABLE && enabled())
                if (state&ST_RESIZING) tiles.draw_tile(surface,10,11,Point(area.w-bsz-tsz,area.h-bsz-tsz));
                else tiles.draw_tile(surface,10,5,Point(area.w-bsz-tsz,area.h-bsz-tsz));
            else tiles.draw_tile(surface,10,8,Point(area.w-bsz-tsz,area.h-bsz-tsz));
        } else if (sbtn()) {
            if (scrollx.enabled() || scrolly.enabled())
                if (has_focus() && !(state&(ST_RESIZING|ST_DRAGGING))) tiles.draw_tile(surface,11,11,Point(area.w-bsz-tsz,area.h-bsz-tsz));
                else tiles.draw_tile(surface,11,5,Point(area.w-bsz-tsz,area.h-bsz-tsz));
            else tiles.draw_tile(surface,11,8,Point(area.w-bsz-tsz,area.h-bsz-tsz));
        }
    }

    bool rbtn() {return bars&RBTN_ALWAYS || ((state&ST_RESIZABLE) && (bars&RBTN_AUTO));}
    bool sbtn() {return !rbtn() && (bars&SBTN_ALWAYS || ((scrollx.visible() && scrolly.visible()) && (bars&SBTN_AUTO)));}

    virtual Surface self() {
        Surface surf(area.w,area.h);
        if (level>=-2 && level <=2)
            get_tileset().draw_frame(surf,6+3*level,0);
        return surf;
    }

    virtual void request_redraw(bool recursive = false) {
        scrollx.request_redraw();
        scrolly.request_redraw();
        Widget::request_redraw(recursive);
    }

    virtual void update_areas() {
        Theme &tiles=get_tileset();
        int tsz = tiles.tilesize;
        int bsz = tiles.bordersize;

        inner_area = {0,0,0,0};

        for (Widget *w : child) inner_area=inner_area|w->area;

        bool needvscroll = inner_area.h>shown_area.h;
        bool needhscroll = inner_area.w>shown_area.w-tsz*needvscroll;
        if ((bars&HBAR_ALWAYS) || (needhscroll && (bars&HBAR_AUTO))) scrollx.show();
        else scrollx.hide();
        if (!(bars&HBAR_ALWAYS_AUTO) || (!needhscroll && (bars&HBAR_AUTO))) scrollx.disable();
        else scrollx.enable();
        if (scrollx.visible()) needvscroll = inner_area.h>area.h-tsz*needhscroll;
        if ((bars&VBAR_ALWAYS) || (needvscroll && (bars&VBAR_AUTO))) scrolly.show();
        else scrolly.hide();
        if (!(bars&VBAR_ALWAYS_AUTO) || (!needvscroll && (bars&VBAR_AUTO))) scrolly.disable();
        else scrolly.enable();

        if (!needhscroll) xoff = 0;
        if (!needvscroll) yoff = 0;

        inner_area.w+=bsz;
        inner_area.h+=bsz;

        inner_area=inner_area|Rect(0,0,area.w-bsz,area.h-bsz);

        shown_area.x=bsz; // Needs to be reset *after* determining which scrollbars to show!
        shown_area.y=bsz;
        shown_area.w=area.w-2*bsz;
        shown_area.h=area.h-2*bsz;

        if (scrollx.visible()) shown_area.h-=tsz;
        if (scrolly.visible()) shown_area.w-=tsz;
    }

    virtual SDL_Rect move_limits() {
        SDL_Rect temp = shown_area;
        temp.x=xoff;
        temp.y=yoff;
        temp.w+=xoff;
        temp.h+=yoff;
        return temp;
    }

    virtual SDL_Point abs_pos() {
        SDL_Point tl = {0,0};
        if (parent) tl = parent->abs_pos();
        tl.x+=area.x+shown_area.x-xoff;
        tl.y+=area.y+shown_area.y-yoff;
        return tl;
    }

    void fix_offsets() {
        int wrange = inner_area.w-shown_area.w;
        int hrange = inner_area.h-shown_area.h;
        if (xoff>wrange) xoff = wrange;
        if (yoff>hrange) yoff = hrange;
        if (xoff<0) xoff = 0;
        if (yoff<0) yoff = 0;
    }

    unsigned bars = HBAR_AUTO|VBAR_AUTO|RBTN_AUTO|SBTN_AUTO;
    enum BARS : unsigned { // Bitfield
        HBAR_NEVER = 0x0,
        HBAR_AUTO = 0x1,
        HBAR_ALWAYS = 0x2,
        HBAR_ALWAYS_AUTO = 0x3,
        VBAR_NEVER = 0x0,
        VBAR_AUTO = 0x4,
        VBAR_ALWAYS = 0x8,
        VBAR_ALWAYS_AUTO = 0xC,
        SBTN_NEVER = 0x00,
        SBTN_AUTO = 0x10,
        SBTN_ALWAYS = 0x20,
        SBTN_ALWAYS_AUTO = 0x30,
        RBTN_NEVER = 0x00,
        RBTN_AUTO = 0x40,
        RBTN_ALWAYS = 0x80,
        RBTN_ALWAYS_AUTO = 0xC0
    };

    HScroll scrollx;
    VScroll scrolly;

    int xoff = 0;
    int yoff = 0;

    SDL_Rect shown_area, inner_area;
};
inline void Widget::addScrollFrame(std::string id, SDL_Rect area, int lvl = 0) throw(Error) {
        get_gui().add(id, new ScrollFrame(this,area,lvl));}

class Label : public Widget {
public:
    Label(Widget *_parent, SDL_Rect _area, const std::string &_caption) :
        Widget(_parent,_area), caption(_caption) {type = TP_LABEL;}

    virtual Surface self() {
        Surface surf(area.w,area.h);
        Theme &t = get_theme();
        Surface text = t.font.write(caption,t.color[enabled()?Theme::TEXT_ENABLED:Theme::TEXT_DISABLED]);
        SDL_Rect dst = text.rect();
        dst.x=t.bordersize+(area.w-dst.w-2*t.bordersize)*h_align;
        dst.y=t.bordersize+(area.h-dst.h-2*t.bordersize)*v_align;
        text.blit(surf,dst);
        return surf;
    }

    std::string caption;
};
inline void Widget::addLabel(std::string id, SDL_Rect area, const std::string &caption) throw(Error) {
        get_gui().add(id, new Label(this,area,caption));}

class Button : public Widget {
public:
    Button(Widget *_parent, SDL_Rect _area, const std::string &_caption, Callback onClick) :
        Widget(_parent,_area), caption(_caption) {type = TP_BUTTON; mouseClick = onClick;}

    virtual bool hit(const SDL_Event &e) {
        if (!enabled()) return Inside(e,area);
        bool consumed = false;
        if (checked() && !sticky) uncheck();
        if (sticky && pressed() && e.type == SDL_MOUSEBUTTONUP && Inside(Point(e.button.x,e.button.y),area)) {
            if (checked()) uncheck();
            else check();
            consumed = true;
        }
        consumed |= Widget::hit(e); // Default stuff
        return consumed;
    }

    virtual Surface self() {
        Theme &t = get_tileset();
        int &bsz = t.bordersize;
        Surface surf(area.w+2*bsz,area.h+2*bsz);
        SDL_Rect inside = Rect(bsz,bsz,area.w,area.h);
        if (enabled()) {
            if (pressed() || checked()) t.draw_frame(surf,0,9,inside);
            else t.draw_frame(surf,0,3,inside);
            if (selected() || highlighted()) t.draw_frame(surf,0,12);
        } else t.draw_frame(surf,0,6,inside);
        Theme &f = get_theme(); // incomplete themes are allowed now
        Surface text = f.font.write(caption,f.color[enabled()?Theme::TEXT_ENABLED:Theme::TEXT_DISABLED]);
        SDL_Rect dst = text.rect();
        dst.x=2*bsz+(area.w-dst.w-2*bsz)*h_align;
        dst.y=2*bsz+(area.h-dst.h-2*bsz)*v_align;
        if (state&(ST_CLICKED|ST_CHECKED)) {
            dst.x++; dst.y++;
        }
        text.blit(surf,dst);
        return surf;
    }

    virtual void draw(Surface target) {
        if (state&ST_VISIBLE) {
            Theme &t = get_tileset();
            int &bsz = t.bordersize;
            SDL_Rect src = area;
            src.x = src.y = 0;
            src.w+=2*bsz; src.h+=2*bsz;
            SDL_Rect dst = area;
            dst.x-=bsz; dst.y-=bsz;
            dst.w+=bsz; dst.h+=bsz;
            surface.blit(target,src,dst);
        }
    }

    std::string caption;
    bool sticky = false;
};
inline void Widget::addButton(std::string id, SDL_Rect area, const std::string &caption, Callback onClick) throw(Error) {
        get_gui().add(id, new Button(this,area,caption,onClick));}

class SimpleButton : public Widget {
public:
    SimpleButton(Widget *_parent, SDL_Rect _area, int _icon_x, int _icon_y, bool _framed, Callback onClick) :
        Widget(_parent,_area), icon_x(_icon_x), icon_y(_icon_y), framed(_framed) {type = TP_SBUTTON; mouseClick = onClick;}

    virtual Surface self() {
        Surface surf(area.w,area.h);
        Theme &t = get_tileset();
        if (framed) {
            int xpos = (area.w-t.tilesize)*h_align;
            int ypos = (area.h-t.tilesize)*v_align;
            if (enabled()) {
                if (pressed() || checked()) {
                    t.draw_frame(surf,0,9);
                    t.draw_tile(surf,icon_x,icon_y+6,Point(xpos,ypos));
                } else {
                    t.draw_frame(surf,0,3);
                    t.draw_tile(surf,icon_x,icon_y,Point(xpos,ypos));
                }
            } else {
                t.draw_frame(surf,0,6);
                t.draw_tile(surf,icon_x,icon_y+3,Point(xpos,ypos));
            }
        } else if (enabled()) {
            if (pressed() || checked()) t.draw_tile(surf,icon_x,icon_y+6,Point(0,0));
            else t.draw_tile(surf,icon_x,icon_y,Point(0,0));
        } else t.draw_tile(surf,icon_x,icon_y+3,Point(0,0));

        return surf;
    }

    int icon_x, icon_y;
    bool framed = false;
};
inline void Widget::addSimpleButton(std::string id, SDL_Rect area, int icon_x, int icon_y, bool framed = false, Callback onClick = nullptr) throw(Error) {
        get_gui().add(id, new SimpleButton(this,area,icon_x,icon_y,framed,onClick));}

class Icon : public Widget {
public:
    Icon(Widget *_parent, SDL_Rect _area, Surface surf, const std::string &_caption = "", Callback onDblClick = nullptr) :
        Widget(_parent,_area), caption(_caption) {type = TP_ICON; surface = surf; mouseDblClick = onDblClick; state|=ST_DRAGGABLE;}

    virtual bool hit(const SDL_Event &e) {
        bool consumed = false;
        if (!enabled()) return Inside(e,area);
        consumed = Widget::hit(e); // Default stuff
        if ((e.type == SDL_MOUSEBUTTONDOWN) && Inside(e,area) && (e.button.button == SDL_BUTTON_LEFT)) {
            check();
            consumed = true;
        }
        if ((e.type == SDL_MOUSEMOTION) && (state&ST_DRAGGABLE) && !(state&ST_DRAGGING) && checked() && (e.motion.state&SDL_BUTTON_LMASK)) {
            take_focus();
            state|=ST_DRAGGING;
            if (mouseDragStart) mouseDragStart(this,e);
            consumed = true;
        }
        if ((e.type == SDL_MOUSEBUTTONDOWN) && !Inside(e,area) && (e.button.button == SDL_BUTTON_LEFT)) {
            uncheck();
            consumed = false;
        }
        return consumed;
    }

    virtual void miss(const SDL_Event &e) {
        Widget::miss(e); // Default stuff
        if ((e.type == SDL_MOUSEBUTTONDOWN) && !Inside(e,area) && (e.button.button == SDL_BUTTON_LEFT)) {
            uncheck();
        }
    }

    virtual bool inject(const SDL_Event &e) {
        if ((!enabled()) || ((e.type == SDL_MOUSEBUTTONDOWN) && !Inside(e,area) && (e.button.button == SDL_BUTTON_LEFT))) {
            lose_focus();
            uncheck();
            request_redraw();
            return false;
        } else if (!(state&ST_DRAGGING)) {
            lose_focus();
            check();
            request_redraw();
            return false;
        }
        return false;
    }

    virtual bool update() {
        if (!visible()) {
            if (state&ST_REDRAW) {
                state&=~ST_REDRAW;
                return true;
            } else return false;
        }
        if (hotkey.mod && my_keymods(SDL_GetModState())) highlight();
        else unhlight(); // Need to do this last minute, because mod keys don't generate SDL_KEYDOWN events
        if (!child.empty()) throw Error("qdw::Icon can't have children!");
        if (state&ST_REDRAW && caption_pos) { // The icon itself is never redrawn/reloaded automatically. That would be too slow and rigid.
            Theme &f = get_theme();
            if (checked()) surf_caption = f.font.write(caption,f.color[Theme::TEXT_SELECTED],f.color[Theme::TEXT_SELECTION]);
            else surf_caption = f.font.write(caption,f.color[enabled()?Theme::TEXT_ENABLED:Theme::TEXT_DISABLED]);
            state&=~ST_REDRAW;
            return true;
        } else return false;
    }

    virtual Surface self() {return surface;} // Should never be used. You can't draw on Icons anyway!

    void make_surfaces(bool s_dis = true, bool s_foc = true, bool s_hlg = false) {
        if (s_dis) {
            surf_disabled = Surface(surface.width(),surface.height());
            for (int y=0; y<surface.height(); y++)
                for (int x=0; x<surface.height(); x++) {
                    SDL_Color c = surface.get_pixel(x,y);
                    Uint8 lum = 64.25f+.5f*luminance(c);
                    surf_disabled.set_pixel(x,y,Color(lum,lum,lum,c.a));
                }
        }
        if (s_foc) {
            surf_focus = Surface(surface.width(),surface.height());
            SDL_Color sel_c = mix(Color(255,255,255,255),get_tileset().color[Theme::TEXT_SELECTION]);
            for (int y=0; y<surface.height(); y++)
                for (int x=0; x<surface.height(); x++) {
                    surf_focus.set_pixel(x,y,sel_c*surface.get_pixel(x,y));
                }
        }
        if (s_hlg) {
            surf_highlight = Surface(surface.width(),surface.height());
            for (int y=0; y<surface.height(); y++)
                for (int x=0; x<surface.height(); x++) {
                    SDL_Color c = surface.get_pixel(x,y);
                    Uint8 alpha = int((128.f+luminance(c))*c.a)>>10;
                    surf_highlight.set_pixel(x,y,Color(255,255,255,alpha));
                }
        }
    }

    virtual void draw(Surface target) {
        if (state&ST_VISIBLE) {
            if (scaleable) {
                if (enabled() || !surf_disabled) {
                    if (checked() && surf_focus) surf_focus.blit_scaled(target,area);
                    else surface.blit_scaled(target,area);
                    if (selected() || highlighted()) surf_highlight.blit_scaled(target,area);
                } else surf_disabled.blit_scaled(target,area);
            } else {
                if (enabled() || !surf_disabled) {
                    if (checked() && surf_focus) surf_focus.blit(target,area);
                    else surface.blit(target,area);
                    if (selected() || highlighted()) surf_highlight.blit(target,area);
                } else surf_disabled.blit(target,area);
            }
        }
        if (surf_caption && caption_pos) {
            Theme &t = get_tileset();
            int &bsz = t.bordersize;
            SDL_Rect dst = surf_caption.rect();
            if (caption_pos&CP_UNDER) {
                dst.x=area.x+bsz+(area.w-dst.w-2*bsz)*h_align;
                dst.y=area.y+area.h+bsz;
                surf_caption.blit(target,dst);
            }
            if (caption_pos&CP_INSIDE) {
                dst.x=area.x+bsz+(area.w-dst.w-2*bsz)*h_align;
                dst.y=area.y+bsz+(area.h-dst.h-2*bsz)*v_align;
                surf_caption.blit(target,dst);
            }
            if (caption_pos&CP_OVER) {
                dst.x=area.x+bsz+(area.w-dst.w-2*bsz)*h_align;
                dst.y=area.y-dst.h;
                surf_caption.blit(target,dst);
            }
            if (caption_pos&CP_RIGHT) {
                dst.x=area.x+area.w+bsz;
                dst.y=area.y+bsz+(area.h-dst.h-2*bsz)*v_align;
                surf_caption.blit(target,dst);
            }
            if (caption_pos&CP_LEFT) {
                dst.x=area.x-dst.w-bsz;
                dst.y=area.y+bsz+(area.h-dst.h-2*bsz)*v_align;
                surf_caption.blit(target,dst);
            }
        }


    }

    unsigned caption_pos = CP_UNDER;
    enum CPOS : unsigned {
        CP_NONE = 0x0,
        CP_UNDER = 0x1,
        CP_INSIDE = 0x2,
        CP_OVER = 0x4,
        CP_RIGHT = 0x8,
        CP_LEFT = 0x10
    };
    bool scaleable = false;
    std::string caption;
    Surface surf_disabled, surf_focus, surf_highlight, surf_caption;
};
inline void Widget::addIcon(std::string id, SDL_Rect area, Surface surf, const std::string &caption = "", Callback onDblClick = nullptr) throw(Error) {
        get_gui().add(id, new Icon(this,area,surf,caption,onDblClick));}

class Grid : public Widget {
public:
    Grid(Widget *_parent, SDL_Rect _area, int cellw, int cellh, int lvl = -1) :
        Widget(_parent,_area), cell({0,0,cellw,cellh}), level(lvl) {type = TP_GRID;}

    virtual Surface self() {
        Surface surf(area.w,area.h);
        if (level>=-2 && level <=2) {
            Theme &t = get_tileset();
            surf.fill(t.color[Theme::WINDOW_BACKGROUND]);
            for (int i = 0; i*cell.w<area.w; i++)
                for (int j = 0; j*cell.h<area.h; j++) {
                    SDL_Rect cc = Rect(i*(cell.x+cell.w)+cell.x*h_align,j*(cell.y+cell.h)+cell.y*v_align,cell.w,cell.h);
                    t.draw_frame(surf,6+3*level,0,cc);
                }
        }
        return surf;
    }

    virtual void handle_drop(Widget *dropped) {
        int gridw = cell.x+cell.w;
        int gridh = cell.y+cell.h;
        SDL_Rect &da = dropped->area;
        da.x-=area.x;
        da.y-=area.y;
        //dropped->change_parent(this);
        switch (adjust) {
        case AJ_ALIGN:
            da.x=int((da.x-cell.x*h_align)/gridw+.5f)*gridw+cell.x*h_align;
            da.y=int((da.y-cell.y*v_align)/gridh+.5f)*gridh+cell.y*v_align;
            break;
        case AJ_CENTER:
            da.x=int((da.x-(cell.w-da.w)/2-cell.x*h_align)/gridw+.5f)*gridw+(cell.w-da.w)/2+cell.x*h_align;
            da.y=int((da.y-(cell.h-da.h)/2-cell.y*v_align)/gridh+.5f)*gridh+(cell.h-da.h)/2+cell.y*v_align;
            break;
        case AJ_SCALE_IND:
            da.x=int((da.x-cell.x*h_align)/gridw+.5f)*gridw+cell.x*h_align;
            da.y=int((da.y-cell.y*v_align)/gridh+.5f)*gridh+cell.y*v_align;
            da.w=cell.w;
            da.h=cell.h;
            break;
        case AJ_SCALE_V:
            da.w*=cell.h/(float)da.h;
            da.h=cell.h;
            da.x=int((da.x-(cell.w-da.w)/2-cell.x*h_align)/gridw+.5f)*gridw+(cell.w-da.w)/2+cell.x*h_align;
            da.y=int((da.y-(cell.h-da.h)/2-cell.y*v_align)/gridh+.5f)*gridh+(cell.h-da.h)/2+cell.y*v_align;
            break;
        case AJ_SCALE_H:
            da.h*=cell.w/(float)da.w;
            da.w=cell.w;
            da.x=int((da.x-(cell.w-da.w)/2-cell.x*h_align)/gridw+.5f)*gridw+(cell.w-da.w)/2+cell.x*h_align;
            da.y=int((da.y-(cell.h-da.h)/2-cell.y*v_align)/gridh+.5f)*gridh+(cell.h-da.h)/2+cell.y*v_align;
            break;
        case AJ_SCALE_MIN:
            {float scale = std::min(cell.w/(float)da.w,cell.h/(float)da.h);
            da.w*=scale;
            da.h*=scale;
            da.x=int((da.x-(cell.w-da.w)/2-cell.x*h_align)/gridw+.5f)*gridw+(cell.w-da.w)/2+cell.x*h_align;
            da.y=int((da.y-(cell.h-da.h)/2-cell.y*v_align)/gridh+.5f)*gridh+(cell.h-da.h)/2+cell.y*v_align;
            }break;
        case AJ_SCALE_MAX:
            {float scale = std::max(cell.w/(float)da.w,cell.h/(float)da.h);
            da.w*=scale;
            da.h*=scale;
            da.x=int((da.x-(cell.w-da.w)/2-cell.x*h_align)/gridw+.5f)*gridw+(cell.w-da.w)/2+cell.x*h_align;
            da.y=int((da.y-(cell.h-da.h)/2-cell.y*v_align)/gridh+.5f)*gridh+(cell.h-da.h)/2+cell.y*v_align;
            }break;
        default:
            break;
        }
        da.x+=area.x;
        da.y+=area.y;
        dropped->request_redraw();
    };

    SDL_Rect cell;
    int level;
    unsigned adjust = AJ_ALIGN;
    enum ADJUST : unsigned {AJ_NONE = 0, AJ_ALIGN, AJ_CENTER, AJ_SCALE_IND, AJ_SCALE_H, AJ_SCALE_V, AJ_SCALE_MIN, AJ_SCALE_MAX};
};
inline void Widget::addGrid(std::string id, SDL_Rect area, int cellw, int cellh, int lvl = -1) throw(Error) {
        get_gui().add(id, new Grid(this,area,cellw,cellh,lvl));}

class CheckBox : public Widget {
public:
    CheckBox(Widget *_parent, SDL_Rect _area, const std::string &_caption) :
        Widget(_parent,_area), caption(_caption) {type = TP_CHECKBOX; h_align=0.f;}

    virtual bool hit(const SDL_Event &e) {
        if (!enabled()) return Inside(e,area);
        bool consumed = false;
        if (pressed() && e.type == SDL_MOUSEBUTTONUP && Inside(Point(e.button.x,e.button.y),area)) {
            if (checked()) uncheck();
            else check();
            consumed = true;
        }
        consumed |= Widget::hit(e); // Default stuff
        return consumed;
    }

    virtual Surface self() {
        Surface surf(area.w,area.h);
        Theme &t = get_tileset();
        t.draw_tile(surf,12, (enabled() ? (pressed()?9:3) : 6) + checked(), Point(0,(area.h-t.tilesize)/2));
        Theme &f = get_theme(); // incomplete themes are allowed now
        Surface text = f.font.write(caption,f.color[enabled()?Theme::TEXT_ENABLED:Theme::TEXT_DISABLED]);
        SDL_Rect dst = text.rect();
        dst.x=t.tilesize+t.bordersize+(area.w-dst.w-t.tilesize-2*t.bordersize)*h_align;
        dst.y=t.bordersize+(area.h-dst.h-2*t.bordersize)*v_align;
        text.blit(surf,dst);
        return surf;
    }

    std::string caption;
};
inline void Widget::addCheckBox(std::string id, SDL_Rect area, const std::string &caption) throw(Error) {
        get_gui().add(id, new CheckBox(this,area,caption));}

class OptionButton : public Widget {
public:
    OptionButton(Widget *_parent, SDL_Rect _area, const std::string &_caption, int _group = 0) :
        Widget(_parent,_area), caption(_caption), group(_group) {type = TP_OPTIONBUTTON; h_align=0.f;}

    virtual bool hit(const SDL_Event &e) {
        if (!enabled()) return Inside(e,area);
        bool consumed = false;
        if (pressed() && e.type == SDL_MOUSEBUTTONUP && Inside(Point(e.button.x,e.button.y),area)) {
            check_only();
            consumed = true;
        }
        consumed |= Widget::hit(e); // Default stuff
        return consumed;
    }

    virtual void check_only() {
        for (Widget *wp : parent->child)
            if ((wp->type==type) && (((OptionButton*)wp)->group==group))
                wp->uncheck();
        check();
    }

    virtual Surface self() {
        Surface surf(area.w,area.h);
        Theme &t = get_tileset();
        t.draw_tile(surf,13, (enabled() ? (pressed()?9:3) : 6) + checked(), Point(0,(area.h-t.tilesize)/2));
        Theme &f = get_theme(); // incomplete themes are allowed now
        Surface text = f.font.write(caption,f.color[enabled()?Theme::TEXT_ENABLED:Theme::TEXT_DISABLED]);
        SDL_Rect dst = text.rect();
        dst.x=t.tilesize+t.bordersize+(area.w-dst.w-t.tilesize-2*t.bordersize)*h_align;
        dst.y=t.bordersize+(area.h-dst.h-2*t.bordersize)*v_align;
        text.blit(surf,dst);
        return surf;
    }

    std::string caption;
    int group;
};
inline void Widget::addOptionButton(std::string id, SDL_Rect area, const std::string &caption, int group = 0) throw(Error) {
        get_gui().add(id, new OptionButton(this,area,caption,group));}

class SubWindow : public ScrollFrame {
public:
    SubWindow(Widget *_parent, SDL_Rect _area, const std::string &_title) :
        ScrollFrame(_parent,_area,2), title(_title),
            btn_min(nullptr, Rect(0,get_tileset().bordersize*3/2,get_tileset().tilesize,get_tileset().tilesize),14,4,false,
                    [&](Widget *,const SDL_Event &){minimize();}),
            btn_max(nullptr, Rect(0,get_tileset().bordersize*3/2,get_tileset().tilesize,get_tileset().tilesize),15,3,false,
                    [&](Widget *,const SDL_Event &){maximize();}),
            btn_norm(nullptr, Rect(0,get_tileset().bordersize*3/2,get_tileset().tilesize,get_tileset().tilesize),14,3,false,
                    [&](Widget *,const SDL_Event &){normalize();}),
            btn_quit(nullptr, Rect(0,get_tileset().bordersize*3/2,get_tileset().tilesize,get_tileset().tilesize),15,4,false,
                    [&](Widget *,const SDL_Event &){close();})
                        {
        btn_min.parent = btn_max.parent = btn_norm.parent = btn_quit.parent = this;
        int &tsz = get_tileset().tilesize;
        int &bsz = get_tileset().bordersize;
        type = TP_SUBWINDOW;
        minw=minh=4*tsz;
        btn_norm.area.x = area.w-4*tsz-bsz-2;
        btn_min.area.x = area.w-3*tsz-bsz-2;
        btn_max.area.x = area.w-2*tsz-bsz-2;
        btn_quit.area.x = area.w-tsz-bsz-1;
        btn_norm.hide(); btn_norm.disable();
        state|=ST_DRAGGABLE|ST_MINABLE|ST_MAXABLE;}

    virtual bool hit(const SDL_Event &e) {
        bool consumed = false;
        if (!enabled()) return Inside(e,area);
        if ((e.type == SDL_MOUSEBUTTONDOWN) && Inside(e,area) && !on_top()) to_top();
        SDL_Point tl = Point(area.x,area.y);
        consumed |= btn_min.hit(e-tl);
        consumed |= btn_max.hit(e-tl);
        consumed |= btn_norm.hit(e-tl);
        consumed |= btn_quit.hit(e-tl);
        if (consumed) return consumed;
        Theme &tiles = get_tileset();
        int &bsz = tiles.bordersize;
        int &tsz = tiles.tilesize;
        if (Inside(e,Rect(area.x+bsz,area.y+bsz,area.w-2*bsz,tsz+bsz))) { // the title bar
            if (e.type == SDL_MOUSEBUTTONDOWN) {
                if (state&ST_DRAGGABLE) {
                    state|=ST_DRAGGING;
                    if (mouseDragStart) mouseDragStart(this,e);
                    take_focus();
                }
                if (e.button.clicks == 2) {
                    if (maximized) normalize();
                    else if (state&ST_MAXABLE) maximize();
                }
            }
            consumed = true;
        }
        consumed |= ScrollFrame::hit(e);
        return consumed;
    }

    virtual bool update() {
        bool ret = false;
        Theme &tiles = get_tileset();
        int &bsz = tiles.bordersize;
        int &tsz = tiles.tilesize;
        if (maximized) {
            area=parent->area;
            area.x=area.y=-bsz;
            area.w+=2*bsz; area.h+=2*bsz;
        }
        btn_min.area.x = area.w-3*tsz-3*bsz;
        btn_max.area.x = area.w-2*tsz-3*bsz;
        btn_quit.area.x = area.w-tsz-2*bsz;
        if (minimized) btn_norm.area.x = btn_min.area.x;
        else btn_norm.area.x = btn_max.area.x;

        ret |= btn_min.update();
        ret |= btn_max.update();
        ret |= btn_quit.update();
        ret |= btn_norm.update();

        if (minimized) {
            //btn_norm.area.x=btn_min.area.x;
            if (state&ST_REDRAW) {
                surface = self();
                ret = true;
                state&=~ST_REDRAW;
            }
        } else ret |= ScrollFrame::update();

        if (btn_min.visible()) btn_min.draw(surface);
        if (btn_max.visible()) btn_max.draw(surface);
        if (btn_norm.visible()) btn_norm.draw(surface);
        if (btn_quit.visible()) btn_quit.draw(surface);

        return ret;
    }

    virtual Surface self() {
        Theme &tiles = get_tileset();
        int &bsz = tiles.bordersize;
        int &tsz = tiles.tilesize;
        Surface surf(area.w,minimized?tsz+3*bsz:area.h);
        tiles.draw_frame(surf,12,0);
        tiles.draw_titlebar(surf,enabled()&&on_top());

        Font &f = tiles.font;
        int oldstyle = f.getStyle();
        f.setStyle(oldstyle|TTF_STYLE_BOLD);
        Surface tsurf = f.write(title,tiles.color[enabled()&&on_top()?Theme::TITLE_ENABLED_TEXT:Theme::TITLE_DISABLED_TEXT]);
        f.setStyle(oldstyle);

        SDL_Rect dst = tsurf.rect();
        dst.x=2*bsz;
        dst.y=bsz+(tsz-dst.h)/2;
        if (icon) {
            icon.blit_scaled(surf,Rect(bsz,bsz,tsz,tsz));
            dst.x+=tsz;
        }
        tsurf.blit(surf,dst);

        return surf;
    }

    virtual void request_redraw(bool recursive = false) {
        btn_min.request_redraw();
        btn_max.request_redraw();
        btn_norm.request_redraw();
        btn_quit.request_redraw();
        ScrollFrame::request_redraw(recursive);
    }

    virtual void update_areas() {
        ScrollFrame::update_areas();
        int &bsz = get_tileset().bordersize;
        int &tsz = get_tileset().tilesize;
        shown_area.y+=tsz+bsz;
        shown_area.h-=tsz+bsz;
    }

    void minimize() {
        if (!minimized && !maximized) normal_area = area;
        Theme &tiles = get_tileset();
        int &bsz = tiles.bordersize;
        int &tsz = tiles.tilesize;
        if (maximized) {
            area.x=normal_area.x;
            area.y=normal_area.y;
            area.w=normal_area.w;
        }
        int tlen = ((4*bsz+4*tsz+(bool)icon*tsz+tiles.font.length(title))/tsz+1)*tsz;
        if (area.w>tlen) area.w=tlen;
        area.h = tsz+3*bsz;
        btn_min.hide(); btn_min.disable();
        btn_max.show();
        if (state&ST_MAXABLE) btn_max.enable();
        btn_norm.show(); btn_norm.enable();
        maximized = false;
        minimized = true;
        request_redraw();
    }

    void maximize() {
        if (!minimized && !maximized) normal_area = area;
        int &bsz = get_tileset().bordersize;
        area=parent->area;
        area.x=area.y=-bsz;
        area.w+=2*bsz; area.h+=2*bsz;
        btn_min.show();
        if (state&ST_MINABLE) btn_min.enable();
        btn_max.hide(); btn_max.disable();
        btn_norm.show(); btn_norm.enable();
        minimized = false;
        maximized = true;
        request_redraw();
    }

    void normalize() {
        if (maximized || minimized) {
            minimized = false;
            maximized = false;
            area = normal_area;
            btn_min.show();
            if (state&ST_MINABLE) btn_min.enable();
            btn_max.show();
            if (state&ST_MAXABLE) btn_max.enable();
            btn_norm.hide(); btn_norm.disable();
            request_redraw();
        }
    }

    void set_button_states(bool mini, bool maxi, bool quit = true) {
        if (mini && !(state&ST_MINABLE)) {
            state|=ST_MINABLE;
            btn_min.enable();
            request_redraw();
        }
        if (!mini && (state&ST_MINABLE)) {
            state&=~ST_MINABLE;
            btn_min.disable();
            request_redraw();
            if (minimized) normalize();
        }
        if (maxi && !(state&ST_MAXABLE)) {
            state|=ST_MAXABLE;
            btn_max.enable();
            request_redraw();
        }
        if (!maxi && (state&ST_MAXABLE)) {
            state&=~ST_MAXABLE;
            btn_max.disable();
            request_redraw();
            if (maximized) normalize();
        }
        if (quit) btn_quit.enable();
        else btn_quit.disable();
    }

    std::string title;
    SimpleButton btn_min,btn_max,btn_norm,btn_quit;
    Surface icon;

private:
    SDL_Rect normal_area;
    bool minimized = false;
    bool maximized = false;
};
inline void Widget::addSubWindow(std::string id, SDL_Rect area, const std::string &title) throw(Error) {
    get_gui().add(id, new SubWindow(this,area,title));}

class EditBox : public Widget {
public:
    EditBox(Widget *_parent, SDL_Rect _area) :
        Widget(_parent,_area) {type = TP_EDITBOX; h_align=0.f;}

    virtual bool hit(const SDL_Event &e) {
        if (!enabled()) return Inside(e,area);
        bool consumed = Widget::hit(e); // Default stuff
        if ((e.type == SDL_MOUSEBUTTONDOWN) && Inside(e,area) && (e.button.button == SDL_BUTTON_LEFT)) {
            take_focus();
            if (e.button.clicks==1) {
                selection = cursor = cursor_pos(e.button.x-area.x);
            } else if (e.button.clicks==2) {
                selection = 0;
                cursor = text.size();
            }
            request_redraw();
        } else if ((e.type == SDL_MOUSEMOTION) && has_focus() && Inside(e,area) && (e.motion.state&SDL_BUTTON_LMASK)) {
            cursor = cursor_pos(e.button.x-area.x);
            request_redraw();
        }
        return consumed;
    }

    virtual bool inject(const SDL_Event &e) {
        bool consumed = false;
        bool changed = false;
        if (!enabled() || ((e.type == SDL_MOUSEBUTTONDOWN) && !Inside(e,area))) {
            lose_focus();
            return consumed;
        }
        if (e.type == SDL_TEXTINPUT && !readonly) {
            delete_selection();
            std::stringstream tmp;
            std::string input(e.text.text);
            int mvcursor = 0;
            if (content&EC_NUMERIC) {
                char c = input[0];
                if (c>='0' && c<='9') input=c;
                else if (c==',' || c=='.') {
                    for (int p = text.size() - 1; p>=0; p--) {
                        if (text[p]=='.') {
                            text.erase(p,1);
                            if (p<cursor) cursor--;
                            if (p<selection) selection--;
                        }
                    }
                    if (!(content&EC_POS_FLOAT) || cursor==(int)text.size() || (cursor==0 && text[0]=='-')) input = "";
                    else input=".";
                }
                else if ((c=='+' || c=='-') && text[0]=='-') {
                    input = "";
                    text = text.substr(1);
                    cursor--;
                } else if (c=='-' && content&EC_NEGATIVE) {
                    input = "";
                    tmp << "-";
                    mvcursor++;
                } else input = "";
            }
            tmp << text.substr(0,cursor) << input << text.substr(cursor);
            text = tmp.str();
            cursor += input.size()+mvcursor;
            selection = cursor;
            changed = true;
            consumed = true;
        }
        if (e.type == SDL_KEYDOWN) {
            bool temp = consumed;
            consumed = true;
            switch (e.key.keysym.sym) {
            case SDLK_LEFT:
                if ((e.key.keysym.mod&KMOD_SHIFT)) move_cursor(-1);
                else {
                    if (selection==cursor) {
                        move_cursor(-1);
                        selection = cursor;
                    } else selection = cursor = std::min(cursor,selection);
                }
                request_redraw();
                break;
            case SDLK_RIGHT:
                if ((e.key.keysym.mod&KMOD_SHIFT)) move_cursor(1);
                else {
                    if (selection==cursor) {
                        move_cursor(1);
                        selection = cursor;
                    } else selection = cursor = std::max(cursor,selection);
                }
                request_redraw();
                break;
            case SDLK_HOME:
                cursor = 0;
                request_redraw();
                break;
            case SDLK_END:
                cursor = text.size();
                request_redraw();
                break;
            case SDLK_DELETE:
                if (readonly) break;
                if (selection==cursor) move_cursor(1);
                delete_selection();
                changed = true;
                break;
            case SDLK_BACKSPACE:
                if (readonly) break;
                if (selection==cursor) move_cursor(-1);
                delete_selection();
                changed = true;
                break;
            case SDLK_RETURN:
            case SDLK_RETURN2:
            case SDLK_KP_ENTER:
                lose_focus();
                request_redraw();
                break;
            default:
                consumed = temp;
            }
        }
        if (changed) request_redraw();
        if (changed && textEdit) textEdit(this,e);
        return consumed;
    }

    virtual void pre_update() {
        if (hotkey.mod && my_keymods(SDL_GetModState())) highlight();
        else unhlight(); // Need to do this last minute, because mod keys don't generate SDL_KEYDOWN events
        if (has_focus()) {
            bool cursor_state = SDL_GetTicks()/get_tileset().cursor_blink%2;
            if (show_cursor!=cursor_state) {
                show_cursor=cursor_state;
                request_redraw();
            }
        }
    }

    int cursor_pos(int xpos) {
        unsigned cp = 0;
        Theme &t = get_tileset();
        Theme &f = get_theme();
        int tlen = f.font.length(text);
        xpos -= 2*t.bordersize+(area.w-tlen-4*t.bordersize)*h_align;
        while (cp<text.size()) {
            if (f.font.length(text.substr(0,cp+1))>xpos) break;
            else cp++;
        }
        return fix_pos(cp);
    }

    virtual Surface self() {
        Surface surf(area.w,area.h);
        Theme &t = get_tileset();
        int &bsz = t.bordersize;
        if (enabled()) {
            if (pressed() || checked()) t.draw_frame(surf,3,9);
            else t.draw_frame(surf,3,3);
            if (selected() || highlighted()) t.draw_frame(surf,3,12,Rect(bsz,bsz,area.w-2*bsz,area.h-2*bsz));
        } else t.draw_frame(surf,3,6);
        Theme &f = get_theme(); // incomplete themes are allowed now
        Surface txt = f.font.write(text,f.color[enabled()?Theme::TEXT_ENABLED:Theme::TEXT_DISABLED]);
        SDL_Rect dst = txt.rect();
        dst.x=2*bsz+(area.w-dst.w-4*bsz)*h_align;
        dst.y=2*bsz+(area.h-dst.h-4*bsz)*v_align;
        txt.blit(surf,dst);
        if (cursor>=0 && selection>=0 && selection!=cursor) {
            int sel_start = std::min(cursor,selection);
            int sel_end = std::max(cursor,selection);
            Surface sel_txt = f.font.write(text.substr(sel_start,sel_end-sel_start),f.color[Theme::TEXT_SELECTED],f.color[Theme::TEXT_SELECTION]);
            SDL_Rect sel_dst = sel_txt.rect();
            sel_dst.x=dst.x+f.font.length(text.substr(0,sel_start));
            sel_dst.y=dst.y;
            sel_txt.blit(surf,sel_dst);
        }
        if (cursor>=0 && show_cursor) t.draw_vbar(surf,6,3,Rect(dst.x+f.font.length(text.substr(0,cursor))-t.tilesize/2,dst.y-t.tilesize,t.tilesize,f.font.height()+2*t.tilesize));
        if (has_focus()) {
            SDL_Point tl = abs_pos();
            SDL_Rect ir = Rect(tl.x+dst.x+f.font.length(text.substr(0,cursor)),tl.y+dst.y,f.font.height(),f.font.height()); // Square should be enough
            SDL_SetTextInputRect(&ir);
        }
        return surf;
    }

    virtual void take_focus() {Widget::take_focus(); if (enabled()) SDL_StartTextInput();};
    virtual void lose_focus() {SDL_StopTextInput(); selection = cursor = -1; Widget::lose_focus();};

    int fix_pos(int p) { // Handles UTF8-encoding
        if (p>0 || p<(int)text.size()) {
            while (p>0 && (text[p]&0xC0)==0x80) p--;
        } else if (p>(int)text.size()) p = text.size();
        else if (p<0) p=0;
        return p;
    }
    void move_cursor(int by = 0) {
        if (by==0) {
            cursor = fix_pos(cursor);
        } else if (by==1) {
            while (cursor<(int)text.size()) if ((text[++cursor]&0xC0)!=0x80) break;
        } else if (by==-1) {
            while (cursor>0) if ((text[--cursor]&0xC0)!=0x80) break;
        } else if (by>1) {
            for (int i=0; i<by; i++) move_cursor(1);
        } else if (by<-1) {
            for (int i=0; i<-by; i++) move_cursor(-1);
        }
    }
    void set_cursor(int where = 0) {
        cursor=0;
        move_cursor(where);
    }
    void delete_selection() {
        int start = fix_pos(std::min(cursor,selection));
        int end = fix_pos(std::max(cursor,selection));
        if (start>=0) {
            std::stringstream tmp;
            tmp << text.substr(0,start) << text.substr(end);
            text = tmp.str();
            selection=cursor=start;
        }
    }

    unsigned content = EC_ANY;
    enum FLAGS : unsigned {
        EC_ANY = 0x0,
        EC_POS_FLOAT = 0x1,
        EC_POS_INTEGER = 0x2,
        EC_NEGATIVE = 0x4,
        EC_FLOAT = 0x5,
        EC_INTEGER = 0x6,
        EC_NUMERIC = 0x7
    };

    int cursor = -1;
    int selection = -1;
    std::string text;
    bool readonly = false;
    bool show_cursor = false;
};
inline void Widget::addEditBox(std::string id, SDL_Rect area) throw(Error) {
        get_gui().add(id, new EditBox(this,area));}

class TextBox : public ScrollFrame {
public:
    TextBox(Widget *_parent, SDL_Rect _area) :
        ScrollFrame(_parent,_area) {type = TP_TEXTBOX; text_area = {0,0,0,0}; h_align=0.f; v_align=0.f; bars = VBAR_AUTO;}

    virtual bool hit(const SDL_Event &e) {
        if (!enabled()) return Inside(e,area);
        bool consumed = ScrollFrame::hit(e); // Default stuff
        if ((e.type == SDL_MOUSEBUTTONDOWN) && Inside(e,shown_area) && (e.button.button == SDL_BUTTON_LEFT)) {
            take_focus();
            if (e.button.clicks==1) {
                selection = cursor = cursor_pos(e.button.x-area.x,e.button.y-area.y);
            } else if (e.button.clicks==2) {
                selection = 0;
                cursor = text.size();
            }
            request_redraw();
        } else if ((e.type == SDL_MOUSEMOTION) && has_focus() && Inside(e,shown_area) && (e.motion.state&SDL_BUTTON_LMASK)) {
            cursor = cursor_pos(e.button.x-area.x,e.button.y-area.y);
            request_redraw();
        }
        return consumed;
    }

    virtual bool inject(const SDL_Event &e) {
        bool consumed = false;
        bool changed = false;
        if (!enabled() || ((e.type == SDL_MOUSEBUTTONDOWN) && !Inside(e,area))) {
            lose_focus();
            return consumed;
        }
        if (!enabled() || (e.type == SDL_MOUSEBUTTONUP)) {
            if (resizing()) {
                state&=~ST_RESIZING;
                if (mouseResizeStop) mouseResizeStop(this,e);
            }
            if (dragging()) {
                state&=~ST_DRAGGING;
                if (mouseDragStop) mouseDragStop(this,e);
            }
            request_redraw();
            return consumed;
        }
        if (e.type == SDL_TEXTINPUT && !readonly) {
            delete_selection();
            std::stringstream tmp;
            std::string input(e.text.text);
            tmp << text.substr(0,cursor) << input << text.substr(cursor);
            text = tmp.str();
            cursor += input.size();
            selection = cursor;
            changed = true;
            consumed = true;
        }
        if (e.type == SDL_KEYDOWN) {
            bool temp = consumed;
            consumed = true;
            switch (e.key.keysym.sym) {
            case SDLK_LEFT:
                if ((e.key.keysym.mod&KMOD_SHIFT)) move_cursor(-1);
                else {
                    if (selection==cursor) {
                        move_cursor(-1);
                        selection = cursor;
                    } else selection = cursor = std::min(cursor,selection);
                }
                request_redraw();
                break;
            case SDLK_RIGHT:
                if ((e.key.keysym.mod&KMOD_SHIFT)) move_cursor(1);
                else {
                    if (selection==cursor) {
                        move_cursor(1);
                        selection = cursor;
                    } else selection = cursor = std::max(cursor,selection);
                }
                request_redraw();
                break;
            case SDLK_UP:
                if ((e.key.keysym.mod&KMOD_SHIFT)) move_line(-1);
                else {
                    if (selection==cursor) {
                        move_line(-1);
                        selection = cursor;
                    } else selection = cursor = std::min(cursor,selection);
                }
                request_redraw();
                break;
            case SDLK_DOWN:
                if ((e.key.keysym.mod&KMOD_SHIFT)) move_line(1);
                else {
                    if (selection==cursor) {
                        move_line(1);
                        selection = cursor;
                    } else selection = cursor = std::max(cursor,selection);
                }
                request_redraw();
                break;
            case SDLK_HOME:
                cursor = 0;
                request_redraw();
                break;
            case SDLK_END:
                cursor = text.size();
                request_redraw();
                break;
            case SDLK_DELETE:
                if (readonly) break;
                if (selection==cursor) move_cursor(1);
                delete_selection();
                changed = true;
                break;
            case SDLK_BACKSPACE:
                if (readonly) break;
                if (selection==cursor) move_cursor(-1);
                delete_selection();
                changed = true;
                break;
            case SDLK_RETURN:
            case SDLK_RETURN2:
            case SDLK_KP_ENTER:
                if (readonly) {
                    lose_focus();
                    request_redraw();
                } else {
                    delete_selection();
                    std::stringstream tmp;
                    tmp << text.substr(0,cursor) << "\n" << text.substr(cursor);
                    text = tmp.str();
                    cursor ++;
                    selection = cursor;
                    changed = true;
                    consumed = true;
                }
                break;
            default:
                consumed = temp;
            }
        }
        if (changed) {
            reset_scroll = true;
            request_redraw();
        }
        if (changed && textEdit) textEdit(this,e);

        return consumed;
    }

    int cursor_pos(int xpos, int ypos) {
        update_lines();
        if (lines.empty()) return 0;
        Theme &t = get_tileset();
        Theme &f = get_theme();
        int cx = (text_area.w-shown_area.w+2*t.bordersize)*scrollx.position;
        int cy = (text_area.h-shown_area.h)*scrolly.position;
        xpos+=cx-2*t.bordersize;
        ypos+=cy-2*t.bordersize;
        int cl = ypos/f.font.lineskip();
        if (cl<0) cl = 0;
        if (cl>=(int)lines.size()) cl = lines.size()-1;
        int ldiff = text_area.w-f.font.length(text.substr(lines[cl].first,lines[cl].second));
        xpos-=ldiff*h_align;
        int cp = 0;
        while (cp<lines[cl].second) {
            if (f.font.length(text.substr(lines[cl].first,cp+1))>xpos) break;
            else cp++;
        }
        return fix_pos(lines[cl].first+cp);
    }

    virtual bool update() {
        if (!visible()) {
            if (state&ST_REDRAW) {
                state&=~ST_REDRAW;
                return true;
            } else return false;
        }
        pre_update();
        for (Widget *w : child) if (w->update()) state|=ST_REDRAW;
        if ((scrollx.state&ST_REDRAW) || (scrolly.state&ST_REDRAW)) state|=ST_REDRAW;
        if (state&ST_REDRAW) {

            update_areas(); // We need to know the line width
            update_text_area(); // To estimate how large the text will be
            update_areas(); // To find out how large our areas really are

            surface = self();
            resize_update();
            draw_text();
            draw_scrollbars();

            state&=~ST_REDRAW;
            return true;
        } else return false;
    }

    void update_text_area() {
        Theme &t = get_tileset();
        Theme &f = get_theme(); // incomplete theme for text
        update_lines();
        text_area.x = 0;
        text_area.y = 0;
        text_area.w = 0;
        text_area.h = lines.size()*f.font.lineskip();
        for (auto &lp : lines) text_area.w=std::max(text_area.w,f.font.length(text.substr(lp.first,lp.second)));
        text_area.w+=2*t.bordersize;
        text_area.h+=2*t.bordersize;
    }

    void draw_text() {
        Theme &t = get_tileset();
        Theme &f = get_theme(); // incomplete theme for text
        update_text_area();
        Surface inner_surf(text_area.w,text_area.h);
        int sel_start = std::min(cursor,selection);
        int sel_end = std::max(cursor,selection);
        int cx = (text_area.w-shown_area.w)*scrollx.position-t.bordersize;
        int cy = (text_area.h-shown_area.h)*scrolly.position;
        for (int i = 0; i<(int)lines.size(); i++) {
            int sls = sel_start;
            if (sls>=0 && sls<lines[i].first) sls = lines[i].first;
            int sll = sel_end-sls;
            if ((sls-lines[i].first)+sll>lines[i].second) sll=lines[i].second-(sls-lines[i].first);
            Surface curline;
            if (sls==lines[i].first && sll>=lines[i].second) { // Select everything
                curline = f.font.write(text.substr(lines[i].first,lines[i].second),f.color[Theme::TEXT_SELECTED],f.color[Theme::TEXT_SELECTION]);
            } else { // Start with unselected text
                curline = f.font.write(text.substr(lines[i].first,lines[i].second),f.color[enabled()?Theme::TEXT_ENABLED:Theme::TEXT_DISABLED]);
                if (sls>=0 && sll>0) { // There's a selection!
                    Surface selsurf = f.font.write(text.substr(sls,sll),f.color[Theme::TEXT_SELECTED],f.color[Theme::TEXT_SELECTION]);
                    selsurf.blit(curline,Rect(f.font.length(text.substr(lines[i].first,sls-lines[i].first)),0,selsurf.width(),selsurf.height()));
                }
            }
            SDL_Rect dst = curline.rect();
            dst.x=(text_area.w-dst.w-2*t.bordersize)*h_align;
            dst.y=t.bordersize+i*f.font.lineskip();
            curline.blit(inner_surf,dst);

            if (cursor>=lines[i].first && cursor<=lines[i].first+lines[i].second) {
                if (show_cursor) t.draw_vbar(inner_surf,6,3,Rect(dst.x+f.font.length(text.substr(lines[i].first,cursor-lines[i].first))-t.tilesize/2,dst.y-t.tilesize,t.tilesize,f.font.height()+2*t.tilesize));
                if (has_focus()) {
                    SDL_Point tl = abs_pos();
                    SDL_Rect ir = Rect(tl.x+dst.x+f.font.length(text.substr(lines[i].first,cursor-lines[i].first))+xoff-cx,tl.y+dst.y+yoff-cy,f.font.height(),f.font.height()); // Square should be enough
                    SDL_SetTextInputRect(&ir);
                }
            }
        }

        inner_surf.blit(surface,Rect(cx,cy,shown_area.w,shown_area.h),shown_area);
    }

    virtual Surface self() {
        Surface surf(area.w,area.h);
        Theme &t = get_tileset();
        if (enabled()) {
            if (pressed() || checked()) t.draw_frame(surf,3,9);
            else t.draw_frame(surf,3,3);
            if (selected() || highlighted()) t.draw_frame(surf,3,12,shown_area);
        } else t.draw_frame(surf,3,6);
        return surf;
    }

    virtual void take_focus() {Widget::take_focus(); if (enabled()) SDL_StartTextInput();};
    virtual void lose_focus() {SDL_StopTextInput(); selection = cursor = -1; Widget::lose_focus();};

    int fix_pos(int p) { // Handles UTF8-encoding
        if (p>0 || p<(int)text.size()) {
            while (p>0 && (text[p]&0xC0)==0x80) p--;
        } else if (p>(int)text.size()) p = text.size();
        else if (p<0) p=0;
        return p;
    }
    void move_cursor(int by = 0) {
        if (by==0) {
            cursor = fix_pos(cursor);
        } else if (by==1) {
            while (cursor<(int)text.size()) if ((text[++cursor]&0xC0)!=0x80) break;
        } else if (by==-1) {
            while (cursor>0) if ((text[--cursor]&0xC0)!=0x80) break;
        } else if (by>1) {
            for (int i=0; i<by; i++) move_cursor(1);
        } else if (by<-1) {
            for (int i=0; i<-by; i++) move_cursor(-1);
        }
    }
    void move_line(int by = 0) {
        if (by!=0) {
            update_lines();
            Theme &f = get_theme();
            int curline=1;
            while (curline<(int)lines.size() && cursor>lines[curline].first) curline++;
            curline--; // A line too far...
            int llen = f.font.length(text.substr(lines[curline].first,cursor-lines[curline].first))-f.font.length("j")/2;
            int tline = curline+by;
            if (tline>=(int)lines.size()) tline = lines.size()-1;
            if (tline<0) tline=0;
            int lpos=0;
            while (lpos<lines[tline].second && f.font.length(text.substr(lines[tline].first,lpos))<llen) lpos++;
            cursor=lines[tline].first+lpos;
        }
        cursor=fix_pos(cursor);
    }
    void set_cursor(int where = 0) {
        cursor=0;
        move_cursor(where);
    }
    void delete_selection() {
        int start = fix_pos(std::min(cursor,selection));
        int end = fix_pos(std::max(cursor,selection));
        if (start>=0) {
            std::stringstream tmp;
            tmp << text.substr(0,start) << text.substr(end);
            text = tmp.str();
            selection=cursor=start;
        }
    }

    virtual void update_areas() {
        Theme &tiles=get_tileset();
        int tsz = tiles.tilesize;
        int bsz = tiles.bordersize;

        inner_area = {0,0,0,0};

        for (Widget *w : child) inner_area=inner_area|w->area;
        inner_area=inner_area|text_area;

        bool needvscroll = inner_area.h>shown_area.h;
        bool needhscroll = inner_area.w>shown_area.w;//-tsz*needvscroll;
        if ((bars&HBAR_ALWAYS) || (needhscroll && (bars&HBAR_AUTO))) scrollx.show();
        else scrollx.hide();
        if (!(bars&HBAR_ALWAYS_AUTO) || (!needhscroll && (bars&HBAR_AUTO))) scrollx.disable();
        else scrollx.enable();
        //if (scrollx.visible()) needvscroll = inner_area.h>area.h-tsz*needhscroll;
        if ((bars&VBAR_ALWAYS) || (needvscroll && (bars&VBAR_AUTO))) scrolly.show();
        else scrolly.hide();
        if (!(bars&VBAR_ALWAYS_AUTO) || (!needvscroll && (bars&VBAR_AUTO))) scrolly.disable();
        else scrolly.enable();

        if (!needhscroll) xoff = 0;
        if (!needvscroll) yoff = 0;

        inner_area.w+=bsz;
        inner_area.h+=bsz;

        inner_area=inner_area|Rect(0,0,area.w-bsz,area.h-bsz);

        shown_area.x=bsz; // Needs to be reset *after* determining which scrollbars to show!
        shown_area.y=bsz;
        shown_area.w=area.w-2*bsz;
        shown_area.h=area.h-2*bsz;

        if (scrollx.visible()) shown_area.h-=tsz;
        if (scrolly.visible()) shown_area.w-=tsz;
    }

    virtual void pre_update() {
        if (hotkey.mod && my_keymods(SDL_GetModState())) highlight();
        else unhlight(); // Need to do this last minute, because mod keys don't generate SDL_KEYDOWN events
        if (has_focus()) {
            bool cursor_state = SDL_GetTicks()/get_tileset().cursor_blink%2;
            if (show_cursor!=cursor_state) {
                show_cursor=cursor_state;
                request_redraw();
            }
        }
    }

    virtual void resize_update() {
        if (reset_scroll) {
            scrollx.position=h_align;
            scrolly.position=v_align;
            reset_scroll=false;
        }
        scrollx.slider=std::min(1.f,shown_area.w/(float)inner_area.w);
        scrolly.slider=std::min(1.f,shown_area.h/(float)inner_area.h);
        scrollx.request_redraw();
        scrolly.request_redraw();
        return;
    }

    void update_lines() {
        lines.clear();
        int start = 0;
        bool full = false;
        int len = -1;
        const int ts = text.size();
        int linewidth = shown_area.w-2*get_tileset().bordersize;
        Theme &f = get_theme();
        while (start<ts) {
            for (int i=start; !full; i++) {
                if (i==ts) {
                    if (!wraplines || (f.font.length(text.substr(start))<linewidth)) len = i-start;
                    break;
                }
                if (wraplines && (text[i]==' ')) {
                    if (f.font.length(text.substr(start,i-start))<linewidth) len=i-start;
                    else {
                        if (len<0) len=i-start;
                        full = true;
                    }
                }
                if ((text[i]=='\n') && !full) {
                    len=i-start;
                    full = true;
                }
            }
            if (len<0) len=ts-start;
            lines.push_back({start,len});
            start+=len+1;
            len=-1;
            full = false;
        }
    }

    int cursor = -1;
    int selection = -1;
    std::string text;
    std::vector<std::pair<int,int>> lines;
    SDL_Rect text_area;
    bool readonly = false;
    bool wraplines = true;
    bool reset_scroll = true;
    bool show_cursor = false;
};
inline void Widget::addTextBox(std::string id, SDL_Rect area) throw(Error) {
        get_gui().add(id, new TextBox(this,area));}

} // namespace qdw

#endif // QDW_HPP
