#include "scene.h"

#include <sstream>
#include <iostream>

GLuint loadtex_sdl(const std::string &filename) {
    static std::map<std::string,GLuint> textures;
    if (!textures.count(filename)) {
        sdw::Surface surf(std::string("test/")+filename);
        textures[filename]=surf.glTexture(true);
    }
    return textures[filename];
}

void Scene::run(sdw::GL_Window &win, sdw::Renderer &ren) {
    bool quit = false;
    sdw::Font arial18("arial.ttf",18);
    int lastframe = SDL_GetTicks();
    std::stringstream temp;

    //Our event structure
    SDL_Event e;
    int counter = 0;
    sdw::Texture ttpf;

    while (!quit) {
        while (SDL_PollEvent(&e)){
            if (e.type == SDL_QUIT)
                quit = true;
            if (e.type == SDL_KEYDOWN){
                switch (e.key.keysym.sym){
                    case SDLK_ESCAPE:
                        quit = true;
                        break;
                    default:
                        break;
                }
            }
        }
        counter++;

        // Performance indicators
        if (0==counter%10) {
            int t = SDL_GetTicks();
            temp.str(""); temp.clear();
            temp<<(t-lastframe)*.1f<<" ms/frm";
            sdw::Surface tpf = arial18.write_solid(temp.str(),sdw::Color(255,0,0));
            ttpf = sdw::Texture(ren,tpf);
            lastframe = t;
        }

        win.fill(0,0,255);
        //win.clear();
        draw();
        ttpf.gl_render(sdw::Point(win.width()-ttpf.width()-5,5));
        win.flip();
    }
}

void Scene::make_terrain() {
    glow::Material::tex_loader = loadtex_sdl;

    glow::VertexDataInfos vdi;
    vdi.push_back(glow::VertexDataInfo(0,GL_VERTEX_ARRAY,3,GL_FLOAT,GL_FALSE,sizeof(VertexC),(void*)offsetof(VertexC,x)));
    vdi.push_back(glow::VertexDataInfo(1,GL_COLOR_ARRAY,4,GL_FLOAT,GL_FALSE,sizeof(VertexC),(void*)offsetof(VertexC,r)));
    terrain = std::make_shared<glow::Geometry<VertexC,GLint,TSTRIP>>(vdi);

#ifdef USE_SHADER
    terrain->make_buffers();
#endif

    // Floor
    terrain->addQuad(
        {-20.0f, -2.0f, 0.0f,      0.f,  .0f,  .0f, 1.f},
        {20.0f, -2.0f, 0.0f,     0.f,  .0f,  .0f, 1.f},
        {20.0f, 50.0f, 0.0f,      0.f,  .0f,  0., 1.f},
        {-20.0f, 50.0f, 0.0f,      0.f,  .0f,  .0f, 1.f});
    // Box
    box(terrain,glm::vec3(-4,-.5f,0),glm::vec3(-2,.5f,1));
    // House
    box(terrain,glm::vec3(-1.1f,3,0),glm::vec3(5.1f,4,5));
    for (int i=-1; i<=5; i+=2) box(terrain,glm::vec3(i-.1f,1,0),glm::vec3(i+.1f,1.2f,4.5f));
    box(terrain,glm::vec3(-1.1f,.8f,2.2f),glm::vec3(5.1f,3.f,2.3f));
    wedge(terrain,glm::vec3(-1.2f,.75f,4.5f),glm::vec3(-1.2f,3.f,5.5f),glm::vec3(-1.2f,3.f,4.5f),6.4f);
    // Tree
    pipe(terrain,glm::vec3(-4.1,3,0),.3f,glm::vec3(-4.05,3,1.5f),.25f);
    cone(terrain,glm::vec3(-4.05,3,1.25f),glm::vec3(-4,3,4.5f),1.25f,13);

    // Stress test: 10 Mio Triangles
    /*for (int x = -16; x<16; x++)
        for (int y = -16; y<16; y++)
            for (int z = 0; z<20; z++) {
                glm::vec4 col(.2f+.2f*glm::sin(x+y*2.67f+z*70.33f+.1234f),
                              .2f+.2f*glm::sin(x*75.5f+y*1.35f+z*6.33f+1.321f),
                              .2f+.2f*glm::sin(x*3.45f+y*40.1f+z*.74f+.2345f),1.f);
                ball(terrain,glm::vec3(x,y,z),.25,25,11,col);
            } */

    mdl.load_mesh("test/sword.obj");

#ifdef USE_SHADER
    terrain->upload();
    terrain->clear_local();

    mdl.upload();
    mdl.clear_local();
#endif

}

struct FogData {float density; glm::vec4 color;};
void Scene::load_shader() {
    shader = std::make_shared<glow::Shader>("persp_color.vs","simple_fog.fs");

    glow::UniformBlock<FogData> fog;
    fog.layout("density",offsetof(FogData,density),sizeof(float));
    fog.layout("fcol",offsetof(FogData,color),sizeof(glm::vec4));
    fog = FogData({0.01f,glm::vec4(0.f,0.f,1.f,1.f)});
    shader->copy_uniform("fog",fog);

    glow::Uniform<glm::mat4> cam = glm::mat4(1.f);
    glow::Uniform<glm::mat4> view = glm::mat4(1.f);
    shader->copy_uniform("MVP",cam);

    glow::Uniform<glm::vec4> ldir = glm::vec4(0.f,0.f,1.f,0.f);
    glow::Uniform<glm::vec4> ambient = glm::vec4(0.05f,0.05f,0.1f,1.f);
    glow::Uniform<glm::vec4> diffuse = glm::vec4(1.f,1.f,0.9f,1.f);

    mdl_shader = std::make_shared<glow::Shader>("persp_tex_norm.vs","material_phong.fs");
    mdl_shader->copy_uniform("MVP",cam);
    mdl_shader->copy_uniform("MV",view);
    mdl_shader->copy_uniform("light_dir",ldir);
    mdl_shader->copy_uniform("light_color",diffuse);
    mdl_shader->copy_uniform("ambient_color",ambient);
    glow::Material::shader = &*mdl_shader;
    glow::Material::texture_target(glow::Material::MAP_NS)=GL_TEXTURE1;
    glow::Material::texture_target(glow::Material::MAP_BUMP)=GL_TEXTURE2;
}

glm::mat4 Scene::set_camera() {
    glm::mat4 View = glm::lookAt(camera,glm::vec3(0,0,(2.f+camera.z)*.5f),glm::vec3(0,0,1));
    return View;
}

void Scene::draw() {
        glClear(GL_DEPTH_BUFFER_BIT);

        glEnable(GL_DEPTH_TEST);
        glEnable(GL_CULL_FACE);

        glm::vec4 ldir = glm::vec4(1,0,1,0);
        glm::vec4 ambient(.1f,.1f,.2f,1.f);
        glm::vec4 diffuse(1.f,.9f,.75f,1.f);

        float t = SDL_GetTicks()*.001f;
        camera = glm::vec3(1,glm::cos(t/8),1.25f+glm::sin(t/8));
#ifdef USE_SHADER
        glm::mat4 Model(1.f);
        glm::mat4 View = set_camera();
        glm::mat4 MVP = Projection*View*Model;
        shader->uniform<glm::mat4>("MVP")=MVP;
        shader->use();

        terrain->draw();

        Model = glm::translate(glm::vec3(0.f,0.f,2.f))*glm::rotate(1.25f*t,glm::vec3(0.f,0.f,1.f));
        MVP = Projection*View*Model;
        glm::mat4 MV = View*Model;

        mdl_shader->uniform<glm::mat4>("MVP")=MVP;
        mdl_shader->uniform<glm::mat4>("MV")=MV;
        mdl_shader->uniform<glm::vec4>("light_dir")=View*ldir;
        mdl_shader->uniform<glm::vec4>("ambient_color")=ambient;
        mdl_shader->uniform<glm::vec4>("light_color")=diffuse;
        mdl_shader->use();

        mdl.draw();

        glUseProgram(0);
#else
        static float old_proj[16], old_mv[16];
        glGetFloatv(GL_PROJECTION_MATRIX,old_proj);
        glGetFloatv(GL_MODELVIEW_MATRIX,old_mv);
        glMatrixMode(GL_PROJECTION);
        glm::mat4 cmat = Projection*set_camera();
        glLoadMatrixf(&cmat[0][0]);
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();

        terrain->draw();

        glEnable(GL_LIGHTING);
        glEnable(GL_LIGHT0);
        glLightfv(GL_LIGHT0, GL_POSITION, &ldir[0]);
        glLightfv(GL_LIGHT0, GL_AMBIENT, &ambient[0]);
        glLightfv(GL_LIGHT0, GL_DIFFUSE, &diffuse[0]);

        glm::mat4 mmat = glm::translate(glm::vec3(0.f,0.f,2.f))*glm::rotate(1.25f*t,glm::vec3(0.f,0.f,1.f));
        glLoadMatrixf(&mmat[0][0]);

        mdl.draw();

        glMatrixMode(GL_MODELVIEW);
        glLoadMatrixf(old_mv);
        glMatrixMode(GL_PROJECTION);
        glLoadMatrixf(old_proj);
        glDisable (GL_LIGHTING);
#endif
        glDisable(GL_TEXTURE_2D);
        glDisable(GL_DEPTH_TEST);
        glDisable(GL_CULL_FACE);
}
