#ifndef GLOW_HPP
#define GLOW_HPP

#include <GL/glew.h>
#include <GL/gl.h>

#include <glm/glm.hpp>

#include <vector>
#include <memory>
#include <string>
#include <map>

#include <fstream> // to load shaders

namespace glow {

struct VertexDataInfo { // gets passed to glVertexAttribPointer()
    VertexDataInfo(GLuint _index,GLenum _content,GLint _size,GLenum _type,GLboolean _norm,GLsizei _stride,GLvoid *_offset) :
        index(_index), content(_content), size(_size), type(_type), norm(_norm), stride(_stride), offset(_offset) {};
    GLuint index;
    GLenum content;
 	GLint size;
 	GLenum type;
 	GLboolean norm;
 	GLsizei stride;
 	GLvoid *offset;
};
typedef std::vector<VertexDataInfo> VertexDataInfos;

template<typename VertexData,typename IndexType = GLuint,bool TriStrips = false> // struct containing vertex data
class Geometry {
public:
    Geometry(const VertexDataInfos &vdi) : vinfo(vdi) {init();}
    Geometry(const VertexDataInfos &vdi, GLenum use) : vinfo(vdi), usage(use), adjust_usage(false) {init();}
    Geometry(const VertexDataInfos &vdi, const std::vector<VertexData> &verts, const std::vector<IndexType> &inds) :
        vinfo(vdi), _vbuf(verts), _ibuf(inds) {init();}

    void make_buffers() {glGenVertexArrays(1,&_abo); glBindVertexArray(_abo);
                         glGenBuffers(1,&_vbo); glGenBuffers(1,&_ibo);}

    ~Geometry() {glDeleteBuffers(1,&_ibo); glDeleteBuffers(1,&_vbo); glDeleteVertexArrays(1,&_abo);}

    unsigned addTri(VertexData v1, VertexData v2, VertexData v3);
    unsigned addTriStrip(std::vector<VertexData> strip);
    unsigned addTriFan(std::vector<VertexData> fan);
    unsigned addQuad(VertexData v1, VertexData v2, VertexData v3, VertexData v4);
    unsigned addPoly(std::vector<VertexData> poly);

    bool on_gpu() {return _abo && _vbo && _ibo;}
    void clear_local() {_vbuf.clear(); _ibuf.clear();} // Free some memory; why have it in both GPU and RAM?
    void clear_gpu() { if (adjust_usage && usage == GL_STATIC_DRAW) usage = GL_STREAM_DRAW; // Obviously not static...
        glBindBuffer(GL_ARRAY_BUFFER,_vbo); glBufferData(GL_ARRAY_BUFFER,0,nullptr,usage);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,_ibo); glBufferData(GL_ELEMENT_ARRAY_BUFFER,0,nullptr,usage); _idx_end=0;}
    void upload();
    void draw() {
        if (on_gpu()) draw_gpu();
        else draw_local();
    };
    void draw_local();
    void draw_gpu();

    VertexDataInfos vinfo;
    GLenum usage = GL_STATIC_DRAW; // May change automatically depending on actual usage,
    bool adjust_usage = true;      // or set by the user and then fixed.
private:
    std::vector<VertexData> _vbuf;
    std::vector<IndexType> _ibuf;
    unsigned _idx_end = 0;
    GLuint _abo = 0;
    GLuint _vbo = 0;
    GLuint _ibo = 0;
    static GLenum _index_type;

    void init() {if (sizeof(IndexType)==1) _index_type = GL_UNSIGNED_BYTE;
                 else if (sizeof(IndexType)==2) _index_type = GL_UNSIGNED_SHORT;
                 else _index_type = GL_UNSIGNED_INT;}
};

namespace _abstract {
    class Uniform {
    public:

    virtual GLuint bind_to(GLuint program, const std::string &name) = 0;
    virtual void upload() = 0;
    virtual void use() = 0;
    operator GLuint() {return _loc;}

    protected:
    Uniform() {};
    GLuint _loc = 0;
    };
}

template<typename T>
class Uniform : public _abstract::Uniform {
public:
    Uniform() {};
    Uniform(const T &unif) {init(unif);}
    const Uniform& operator=(const T &unif) {if ((bool)_u) *_u=unif; else init(unif); return *this;}

    virtual GLuint bind_to(GLuint program, const std::string &varname) {return _loc = glGetUniformLocation(program, varname.c_str());}
    virtual T& val() {return *_u;}
    virtual void upload();
    virtual void use() {upload();};

private:
    void init(const T &unif) {_u = std::make_shared<T>(unif);}
    std::shared_ptr<T> _u;
};

typedef std::map<std::string,std::pair<size_t,size_t>> StructLayout;

template<typename T>
class UniformBlock : public _abstract::Uniform {
public:
    UniformBlock() {}
    UniformBlock(const T &unif_struct) {init(unif_struct);}
    const UniformBlock& operator=(const T &unif_struct) {if ((bool)_u) *_u=unif_struct; else init(unif_struct); return *this;}

    void layout(const StructLayout &vars) {_offsets = vars;}
    void layout(const std::string &varname, size_t offset, size_t varsize) {
        _offsets[varname]=std::make_pair(offset,varsize);}
    virtual GLuint bind_to(GLuint program, const std::string &blockname) {_prog = program; return _loc = glGetUniformBlockIndex(program,blockname.c_str());}
    virtual T& val() {return *_u;}
    virtual void upload();
    void upload(const std::string &varname);
    virtual void use() {glBindBufferBase(GL_UNIFORM_BUFFER,_loc,_buf);}

private:
    void init(const T &unif_struct) {_u = std::make_shared<T>(unif_struct); glGenBuffers(1,&_buf);}
    GLuint _buf = 0;
    GLuint _prog = 0;
    std::shared_ptr<T> _u;
    StructLayout _offsets;
};

typedef std::shared_ptr<_abstract::Uniform> UniformP;
template <typename U>
UniformP uniform_ptr(const U &u) {return std::make_shared<U>(u);};

class Shader {
public:
    Shader() {};
    Shader(const std::string &vertex_shader, const std::string &fragment_shader) {
        add(vertex_shader,GL_VERTEX_SHADER);
        add(fragment_shader,GL_FRAGMENT_SHADER);
        link(); clear_shaders();
    };
    Shader(const std::string &vertex_shader, const std::string &geometry_shader, const std::string &fragment_shader) {
        add(vertex_shader,GL_VERTEX_SHADER);
        add(geometry_shader,GL_GEOMETRY_SHADER);
        add(fragment_shader,GL_FRAGMENT_SHADER);
        link(); clear_shaders();
    };
    ~Shader() {clear_shaders(); clear_program();}

    operator GLuint() {return _prog;}

    GLuint add(const std::string &path, GLenum type) throw(std::string);
    GLuint link() throw(std::string);
    void use() {glUseProgram(_prog); for (auto &u : _unifs) u.second->use();}
    void clear_shaders() {for (GLuint sid : _shaders) glDeleteShader(sid); _shaders.clear();}
    void clear_program() {glDeleteProgram(_prog);}

    template <typename U>
    void copy_uniform(const std::string &varname, const U &unif) {bind_uniform(varname,uniform_ptr(unif));}
    void bind_uniform(const std::string &varname, UniformP unif) {_unifs[varname]=unif; unif->bind_to(_prog,varname); unif->upload();}
    void clear_uniform(const std::string &varname) {_unifs.erase(varname);}
    void clear_uniforms() {_unifs.clear();}
    void upload_uniform(const std::string &varname) {_unifs[varname]->upload();}

    template <typename U>
    U &uniform(const std::string &varname) {return static_cast<Uniform<U>*>(&*_unifs[varname])->val();}
    template <typename U>
    U &uniform_block(const std::string &varname) {return static_cast<UniformBlock<U>*>(&*_unifs[varname])->val();}

private:
    GLuint _prog = 0;
    std::vector<GLuint> _shaders;
    std::map<std::string,UniformP> _unifs;
};



template<typename VertexData,typename IndexType,bool TriStrips>
GLenum Geometry<VertexData,IndexType,TriStrips>::_index_type;

template<typename VertexData,typename IndexType,bool TriStrips>
unsigned Geometry<VertexData,IndexType,TriStrips>::addTri(VertexData v1, VertexData v2, VertexData v3) {
    unsigned start = _vbuf.size();
    _vbuf.push_back(v1);
    _vbuf.push_back(v2);
    _vbuf.push_back(v3);
    if (TriStrips) { // This is a compile-time constant; it will be optimized out.
        _ibuf.push_back(start); // "stutter" at beginning and end to degenerate adjacent triangles
        _ibuf.push_back(start);
        _ibuf.push_back(start+1);
        _ibuf.push_back(start+2);
        _ibuf.push_back(start+2);
        _ibuf.push_back(start+2); // Total number of vertices must be even!
    } else {
        _ibuf.push_back(start);
        _ibuf.push_back(start+1);
        _ibuf.push_back(start+2);
    }
    return start;
}
template<typename VertexData,typename IndexType,bool TriStrips>
unsigned Geometry<VertexData,IndexType,TriStrips>::addTriStrip(std::vector<VertexData> strip) {
    unsigned start = _vbuf.size();
    if (strip.size()>2) {
        _vbuf.insert(_vbuf.end(),strip.begin(),strip.end());
        if (TriStrips) {
            _ibuf.push_back(start);
            for (unsigned i=0; i<strip.size(); i++) _ibuf.push_back(start+i);
            _ibuf.push_back(start+strip.size()-1); // Stutter at the end
            if (strip.size()%2) _ibuf.push_back(start+strip.size()-1); // Even number of vertices, to keep correct facing
        } else {
            for (unsigned i=2; i<strip.size(); i++) if (i%2) {
                _ibuf.push_back(start+i);
                _ibuf.push_back(start+i-1);
                _ibuf.push_back(start+i-2);
            } else {
                _ibuf.push_back(start+i-2);
                _ibuf.push_back(start+i-1);
                _ibuf.push_back(start+i);
            }
        }
    }
    return start;
}
template<typename VertexData,typename IndexType,bool TriStrips>
unsigned Geometry<VertexData,IndexType,TriStrips>::addTriFan(std::vector<VertexData> fan) {
    unsigned start = _vbuf.size();
    if (fan.size()>2) {
        _vbuf.insert(_vbuf.end(),fan.begin(),fan.end());
        if (TriStrips) {
            _ibuf.push_back(start+1);
            for (unsigned i=1; i<fan.size()-1; i+=2) {
                _ibuf.push_back(start+i);
                _ibuf.push_back(start+i); // Many degenerate triangles here, but those are cheap...
                _ibuf.push_back(start);
                _ibuf.push_back(start+i+1);
            }
            if (fan.size()%2==0) {
                _ibuf.push_back(start+fan.size()-1);
                _ibuf.push_back(start+fan.size()-1);
                _ibuf.push_back(start);
                _ibuf.push_back(start+1);
            } else {
                _ibuf.push_back(start+1);
                _ibuf.push_back(start+1);
            }
            _ibuf.push_back(start+1);
        } else {
            for (unsigned i=1; i<fan.size(); i++) {
                _ibuf.push_back(start);
                _ibuf.push_back(start+i-1);
                _ibuf.push_back(start+i);
            }
            _ibuf.push_back(start);
            _ibuf.push_back(start+fan.size()-1);
            _ibuf.push_back(start+1);
        }
    }
    return start;
}
template<typename VertexData,typename IndexType,bool TriStrips>
unsigned Geometry<VertexData,IndexType,TriStrips>::addQuad(VertexData v1, VertexData v2, VertexData v3, VertexData v4) {
    unsigned start = _vbuf.size();
    _vbuf.push_back(v1);
    _vbuf.push_back(v2);
    _vbuf.push_back(v3);
    _vbuf.push_back(v4);
    if (TriStrips) {
        _ibuf.push_back(start);
        _ibuf.push_back(start);
        _ibuf.push_back(start+1);
        _ibuf.push_back(start+3);
        _ibuf.push_back(start+2);
        _ibuf.push_back(start+2);
    } else {
        _ibuf.push_back(start);
        _ibuf.push_back(start+1);
        _ibuf.push_back(start+3);
        _ibuf.push_back(start+1);
        _ibuf.push_back(start+2);
        _ibuf.push_back(start+3);
    }
    return start;
}
template<typename VertexData,typename IndexType,bool TriStrips>
unsigned Geometry<VertexData,IndexType,TriStrips>::addPoly(std::vector<VertexData> poly) {
    unsigned start = _vbuf.size();
    if (poly.size()>3) {
        _vbuf.insert(_vbuf.end(),poly.begin(),poly.end());
        unsigned i,j;
        if (TriStrips) {
            _ibuf.push_back(start);
            _ibuf.push_back(start); // start degenerate
            for (i=1, j=poly.size()-1; i<j; i++,j--) {
                _ibuf.push_back(start+i);
                _ibuf.push_back(start+j);
            }
            unsigned last = start+(poly.size()+1)/2;
            _ibuf.push_back(last);
            _ibuf.push_back(last);
        } else {
            unsigned lt=0;
            unsigned lb=1;
            for (i=2, j=poly.size()-1; i<j; i++,j--) {
                _ibuf.push_back(start+lt);
                _ibuf.push_back(start+lb);
                _ibuf.push_back(start+j);
                _ibuf.push_back(start+lb);
                _ibuf.push_back(start+i);
                _ibuf.push_back(start+j);
                lt=j; lb=i;
            }
            if (i==j) {
                unsigned last = start+(poly.size()+1)/2;
                _ibuf.push_back(last-1);
                _ibuf.push_back(last);
                _ibuf.push_back(last+1);
            }
        }
    } else if (poly.size()==3) addTri(poly[0],poly[1],poly[2]);
    return start;
}

template<typename VertexData,typename IndexType,bool TriStrips>
void Geometry<VertexData,IndexType,TriStrips>::upload() {
    static bool first = true;
    if (adjust_usage && !first && usage == GL_STATIC_DRAW) usage = GL_DYNAMIC_DRAW; // called a second update() before clear_local()
    first = false;

    glBindVertexArray(_abo);

    glBindBuffer(GL_ARRAY_BUFFER, _vbo);
    glBufferData(GL_ARRAY_BUFFER, _vbuf.size()*sizeof(VertexData), &_vbuf[0], usage);
    for (auto &vi : vinfo) glVertexAttribPointer(vi.index,vi.size,vi.type,vi.norm,vi.stride,vi.offset);

    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _ibo);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, (_ibuf.size()-TriStrips)*sizeof(IndexType), &_ibuf[TriStrips], usage);

    _idx_end=_ibuf.size();
}
template<typename VertexData,typename IndexType,bool TriStrips>
void Geometry<VertexData,IndexType,TriStrips>::draw_gpu() {
    glBindVertexArray(_abo);

    glBindBuffer(GL_ARRAY_BUFFER, _vbo);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER,_ibo);

    for (auto &vi : vinfo) glEnableVertexAttribArray(vi.index);

    glDrawElements(TriStrips?GL_TRIANGLE_STRIP:GL_TRIANGLES,_idx_end,_index_type,nullptr);

    for (auto &vi : vinfo) glDisableVertexAttribArray(vi.index);
}
template<typename VertexData,typename IndexType,bool TriStrips>
void Geometry<VertexData,IndexType,TriStrips>::draw_local() {
    char *mempos = (char*)&_vbuf[0];
    for (auto &vi : vinfo) {
        switch (vi.content) {
        case GL_VERTEX_ARRAY:
            glEnableClientState(GL_VERTEX_ARRAY);
            glVertexPointer(vi.size, vi.type, vi.stride, mempos+(int)vi.offset);
            break;
        case GL_TEXTURE_COORD_ARRAY:
            glEnableClientState(GL_TEXTURE_COORD_ARRAY);
            glTexCoordPointer(vi.size, vi.type, vi.stride, mempos+(int)vi.offset);
            break;
        case GL_NORMAL_ARRAY:
            glEnableClientState(GL_NORMAL_ARRAY);
            glNormalPointer(vi.type, vi.stride, mempos+(int)vi.offset);
            break;
        case GL_COLOR_ARRAY:
            glEnableClientState(GL_COLOR_ARRAY);
            glColorPointer(vi.size, vi.type, vi.stride, mempos+(int)vi.offset);
            break;
        case GL_SECONDARY_COLOR_ARRAY:
            glEnableClientState(GL_SECONDARY_COLOR_ARRAY);
            glColorPointer(vi.size, vi.type, vi.stride, mempos+(int)vi.offset);
            break;
        case GL_INDEX_ARRAY: // Color index!
            glEnableClientState(GL_INDEX_ARRAY);
            glIndexPointer(vi.type, vi.stride, mempos+(int)vi.offset);
            break;
        case GL_EDGE_FLAG_ARRAY:
            glEnableClientState(GL_EDGE_FLAG_ARRAY);
            glEdgeFlagPointer(vi.stride, mempos+(int)vi.offset);
            break;
        case GL_FOG_COORD_ARRAY:
            glEnableClientState(GL_FOG_COORD_ARRAY);
            glFogCoordPointer(vi.type, vi.stride, mempos+(int)vi.offset);
            break;
        default:
            continue;
        }
    }
    glDrawElements(TriStrips?GL_TRIANGLE_STRIP:GL_TRIANGLES,_ibuf.size()-TriStrips,_index_type,&_ibuf[TriStrips]);

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);
    glDisableClientState(GL_COLOR_ARRAY);
    glDisableClientState(GL_SECONDARY_COLOR_ARRAY);
    glDisableClientState(GL_INDEX_ARRAY);
    glDisableClientState(GL_FOG_COORD_ARRAY);
    glDisableClientState(GL_EDGE_FLAG_ARRAY);
}

// Uniforms
// OpenGL uses a different function for each type, so there's a lot of functions here...
template <> inline void Uniform<GLint>::upload() {glUniform1i(_loc,*_u);}
template <> inline void Uniform<GLuint>::upload() {glUniform1ui(_loc,*_u);}
template <> inline void Uniform<GLfloat>::upload() {glUniform1f(_loc,*_u);}
template <> inline void Uniform<glm::vec2>::upload() {glUniform2f(_loc,_u->x,_u->y);}
template <> inline void Uniform<glm::vec3>::upload() {glUniform3f(_loc,_u->x,_u->y,_u->z);}
template <> inline void Uniform<glm::vec4>::upload() {glUniform4f(_loc,_u->x,_u->y,_u->z,_u->w);}
template <> inline void Uniform<glm::mat2>::upload() {glUniformMatrix2fv(_loc,1,GL_FALSE,&(*_u)[0][0]);}
template <> inline void Uniform<glm::mat3>::upload() {glUniformMatrix3fv(_loc,1,GL_FALSE,&(*_u)[0][0]);}
template <> inline void Uniform<glm::mat4>::upload() {glUniformMatrix4fv(_loc,1,GL_FALSE,&(*_u)[0][0]);}
template <> inline void Uniform<glm::mat2x3>::upload() {glUniformMatrix2x3fv(_loc,1,GL_FALSE,&(*_u)[0][0]);}
template <> inline void Uniform<glm::mat3x2>::upload() {glUniformMatrix3x2fv(_loc,1,GL_FALSE,&(*_u)[0][0]);}
template <> inline void Uniform<glm::mat2x4>::upload() {glUniformMatrix2x4fv(_loc,1,GL_FALSE,&(*_u)[0][0]);}
template <> inline void Uniform<glm::mat4x2>::upload() {glUniformMatrix4x2fv(_loc,1,GL_FALSE,&(*_u)[0][0]);}
template <> inline void Uniform<glm::mat3x4>::upload() {glUniformMatrix3x4fv(_loc,1,GL_FALSE,&(*_u)[0][0]);}
template <> inline void Uniform<glm::mat4x3>::upload() {glUniformMatrix4x3fv(_loc,1,GL_FALSE,&(*_u)[0][0]);}
// std::vector is guaranteed to be contiguous
template <> inline void Uniform<std::vector<GLint>>::upload() {glUniform1iv(_loc,_u->size(),&_u->front());}
template <> inline void Uniform<std::vector<GLuint>>::upload() {glUniform1uiv(_loc,_u->size(),&_u->front());}
template <> inline void Uniform<std::vector<GLfloat>>::upload() {glUniform1fv(_loc,_u->size(),&_u->front());}
// glm::vec* relies on implementation details; this might fail on some obscure compilers/systems
template <> inline void Uniform<std::vector<glm::vec2>>::upload() {glUniform2fv(_loc,_u->size(),(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::vec3>>::upload() {glUniform3fv(_loc,_u->size(),(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::vec4>>::upload() {glUniform4fv(_loc,_u->size(),(GLfloat*)&_u->front());}
// This stuff makes a lot of assumptions about your compiler. May or may not work.
template <> inline void Uniform<std::vector<glm::mat2>>::upload() {glUniformMatrix2fv(_loc,_u->size(),GL_FALSE,(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::mat3>>::upload() {glUniformMatrix3fv(_loc,_u->size(),GL_FALSE,(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::mat4>>::upload() {glUniformMatrix4fv(_loc,_u->size(),GL_FALSE,(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::mat2x3>>::upload() {glUniformMatrix2x3fv(_loc,_u->size(),GL_FALSE,(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::mat3x2>>::upload() {glUniformMatrix3x2fv(_loc,_u->size(),GL_FALSE,(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::mat2x4>>::upload() {glUniformMatrix2x4fv(_loc,_u->size(),GL_FALSE,(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::mat4x2>>::upload() {glUniformMatrix4x2fv(_loc,_u->size(),GL_FALSE,(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::mat3x4>>::upload() {glUniformMatrix3x4fv(_loc,_u->size(),GL_FALSE,(GLfloat*)&_u->front());}
template <> inline void Uniform<std::vector<glm::mat4x3>>::upload() {glUniformMatrix4x3fv(_loc,_u->size(),GL_FALSE,(GLfloat*)&_u->front());}

template <typename T>
void UniformBlock<T>::upload() {
    if (_offsets.empty()) return;
    GLint bsz;
    glGetActiveUniformBlockiv(_prog,_loc,GL_UNIFORM_BLOCK_DATA_SIZE,&bsz);
    glBindBuffer(GL_UNIFORM_BUFFER,_buf);
    glBufferData(GL_UNIFORM_BUFFER,bsz,0,GL_DYNAMIC_DRAW);
    for (auto &var : _offsets) {
        GLuint index;
        const GLchar *sptr = var.first.c_str();
        glGetUniformIndices(_prog,1,&sptr,&index);
        GLint offset;
        glGetActiveUniformsiv(_prog,1,&index,GL_UNIFORM_OFFSET,&offset);
        if (offset+(int)var.second.second<=bsz)
            glBufferSubData(GL_UNIFORM_BUFFER,offset,var.second.second,(char*)&*_u+var.second.first);
    }
}
template <typename T>
void UniformBlock<T>::upload(const std::string &varname) {
    if (!_offsets.count(varname)) return;
    GLint bsz;
    glGetActiveUniformBlockiv(_prog,_loc,GL_UNIFORM_BLOCK_DATA_SIZE,&bsz);
    GLuint index;
    const GLchar *sptr = varname.c_str();
    glGetUniformIndices(_prog,1,&sptr,&index);
    GLint offset;
    glGetActiveUniformsiv(_prog,1,&index,GL_UNIFORM_OFFSET,&offset);
    auto memloc = _offsets[varname];
    if (offset+memloc.second<=bsz) {
        glBindBuffer(GL_UNIFORM_BUFFER,_buf);
        glBufferSubData(GL_UNIFORM_BUFFER,offset,memloc.second,(char*)&*_u+memloc.first);
    }
}

inline GLuint Shader::add(const std::string &path, GLenum type) throw(std::string) {
    GLuint id = glCreateShader(type);

    std::string code;
    std::ifstream fs(path, std::ios::in);
    if(fs.is_open()) {
        std::string line;
        while(getline(fs, line)) code += "\n" + line;
        fs.close();
    } else {
        glDeleteShader(id);
        throw(std::string("Failed to load shader \"")+path+"\".\n");
    }

    GLint ok = GL_FALSE;

    char const *source = code.c_str();
    glShaderSource(id, 1, &source , NULL);
    glCompileShader(id);

    glGetShaderiv(id, GL_COMPILE_STATUS, &ok);
    if (GL_FALSE == ok) {
        int logsz;
        glGetShaderiv(id, GL_INFO_LOG_LENGTH, &logsz);
        std::vector<char> error(logsz);
        glGetShaderInfoLog(id, logsz, NULL, &error[0]);
        glDeleteShader(id);
        throw(std::string("Failed to compile shader \"")+path+"\":\n"+std::string(&error[0]));
    }

    _shaders.push_back(id);
    return id;
}

inline GLuint Shader::link() throw(std::string) {
    GLuint id = glCreateProgram();

    for (GLuint sid : _shaders) glAttachShader(id, sid);
    glLinkProgram(id);

    GLint ok = GL_FALSE;

    glGetProgramiv(id, GL_LINK_STATUS, &ok);
    if (GL_FALSE == ok) {
        int logsz;
        glGetProgramiv(id, GL_INFO_LOG_LENGTH, &logsz);
        std::vector<char> error(std::max(logsz, int(1)));
        glGetProgramInfoLog(id, logsz, NULL, &error[0]);
        glDeleteProgram(id);
        throw(std::string("Failed to link program:\n")+std::string(&error[0]));
    }

    if (_prog) glDeleteProgram(_prog);
    _prog = id;
    return id;
}

} // namespace glow
#endif // GLOW_HPP
