#include "glow_wavefront.hpp"

namespace glow {

Shader *Material::shader = nullptr;
TextureLoader Material::tex_loader;
GLenum Material::_tex_tar[MAP_MAX] = {GL_TEXTURE0,GL_TEXTURE0,0,0,0,0,0,0};
std::map<std::string,Material> Material::materials;

namespace {
    struct Index3 {int v=0; int t=0; int n=0;};
    bool operator<(const Index3 &i1, const Index3 &i2) {
        if (i1.v<i2.v) return true;
        if (i1.v>i2.v) return false;
        if (i1.t<i2.t) return true;
        if (i1.t>i2.t) return false;
        if (i1.n<i2.n) return true;
        return false;
    };
} // anonymous namespace

void Object::load_mesh(const std::string &path, bool flip_v) {
    // Vectors to store model data
    std::vector<glm::vec4> verts({glm::vec4(0.f,0.f,0.f,1.f)});
    std::vector<glm::vec3> texcoords({glm::vec3(0.f,0.f,0.f)});
    std::vector<glm::vec3> norms({glm::vec3(0.f,0.f,0.f)});
    std::vector<std::vector<Index3>> faces;
    std::vector<std::pair<int,std::string>> mats;
    // Read model data from file
    std::ifstream fs(path, std::ios::in);
    if(fs.is_open()) {
        std::string line, token;
        std::stringstream lstr;
        auto lastline = fs.tellg();
        while(getline(fs, line)) {
            lstr.clear();
            lstr.str(line);
            lstr>>token;
            if (token=="v") {
                glm::vec4 temp(verts[0]);
                if (lstr.good()) lstr>>temp.x;
                if (lstr.good()) lstr>>temp.y;
                if (lstr.good()) lstr>>temp.z;
                if (lstr.good()) lstr>>temp.w;
                verts.push_back(temp);
            } else if (token=="vt") {
                glm::vec3 temp(texcoords[0]);
                if (lstr.good()) lstr>>temp.x;
                if (lstr.good()) lstr>>temp.y;
                if (lstr.good()) lstr>>temp.z;
                if (flip_v) temp.y=1.f-temp.y;
                texcoords.push_back(temp);
            } else if (token=="vn") {
                glm::vec3 temp(norms[0]);
                if (lstr.good()) lstr>>temp.x;
                if (lstr.good()) lstr>>temp.y;
                if (lstr.good()) lstr>>temp.z;
                temp=glm::normalize(temp);
                norms.push_back(temp);
            } else if (token=="f") {
                std::vector<Index3> tempF;
                std::string tempV;
                while (lstr.good()) {
                    tempV="";
                    lstr>>tempV;
                    Index3 i3;
                    std::stringstream vstream(tempV);
                    if (vstream.good()) vstream>>i3.v;
                    if (vstream.good() && vstream.peek()=='/') vstream.get();
                    if (vstream.good() && vstream.peek()!='/') vstream>>i3.t;
                    if (vstream.good() && vstream.peek()=='/') vstream.get();
                    if (vstream.good() && vstream.peek()!='/') vstream>>i3.n;
                    if (i3.v<0) i3.v=(int)verts.size()+i3.v;
                    if (i3.t<0) i3.t=(int)texcoords.size()+i3.t;
                    if (i3.n<0) i3.n=(int)norms.size()+i3.n;
                    if ((i3.v>0)&&(i3.t>0)&&(i3.n>0))tempF.push_back(i3);
                }
                faces.push_back(tempF);
            } else if (token=="usemtl") {
                std::string temp;
                lstr>>temp;
                mats.push_back(std::make_pair((int)faces.size(),temp));
            } else if (token=="newmtl") {
                fs.seekg(lastline);
                Material::read(fs);
            } else if (token=="mtllib") {
                std::string temp;
                lstr>>temp;
                int lastslash;
                for (int i=0; i<(int)path.size(); i++) if (path[i]=='/' || path[i]=='\\') lastslash=i;
                temp = path.substr(0,lastslash+1)+temp;
                if (!(temp.substr(temp.size()-4,temp.size())==".mtl")) temp+=".mtl";
                Material::load(temp);
            }
            token.clear();
            lastline = fs.tellg();
        }
        fs.close();
    } // File is read now
    // And put it into a format OpenGL can use
    std::vector<ObjVertex> vstructs;
    for (auto p = mats.begin(); p!=mats.end(); p++) {
        int s_ind = p->first;
        int e_ind = (p+1==mats.end())?faces.size():(p+1)->first;
        Submesh smesh(p->second);
        for (int i=s_ind; i<e_ind; i++) {
            auto &face = faces[i];
            vstructs.clear();
            for (auto &i3 : face) {
                ObjVertex ov;
                ov.x=verts[i3.v].x;
                ov.y=verts[i3.v].y;
                ov.z=verts[i3.v].z;
                ov.w=verts[i3.v].w;
                ov.tu=texcoords[i3.t].x;
                ov.tv=texcoords[i3.t].y;
                ov.tw=texcoords[i3.t].z;
                ov.nx=norms[i3.n].x;
                ov.ny=norms[i3.n].y;
                ov.nz=norms[i3.n].z;
                vstructs.push_back(ov);
            }
            smesh.mesh.addPoly(vstructs);
        }
        _meshes.push_back(smesh);
    }
};

void Material::read(std::ifstream &file) {
    std::string line, token;
    std::stringstream lstr;
    auto lastline = file.tellg();
    Material cur_mat;
    std::string matname = ""; // Yup, this will occasionally overwrite Material[""] with the default. It's not a bug, it's a feature.
    while(getline(file, line)) {
        lstr.clear();
        lstr.str(line);
        lstr>>token;
        if (token=="newmtl") {
            materials[matname]=cur_mat;
            lstr>>matname;
        } else if (token=="Ka") {
            if (lstr.good()) lstr>>cur_mat.Ka.r;
            if (lstr.good()) lstr>>cur_mat.Ka.g;
            if (lstr.good()) lstr>>cur_mat.Ka.b;
        } else if (token=="Kd") {
            if (lstr.good()) lstr>>cur_mat.Kd.r;
            if (lstr.good()) lstr>>cur_mat.Kd.g;
            if (lstr.good()) lstr>>cur_mat.Kd.b;
        } else if (token=="Ks") {
            if (lstr.good()) lstr>>cur_mat.Ks.r;
            if (lstr.good()) lstr>>cur_mat.Ks.g;
            if (lstr.good()) lstr>>cur_mat.Ks.b;
        } else if (token=="Ns") {
            if (lstr.good()) lstr>>cur_mat.Ns;
        } else if (token=="Tr" || token=="d") {
            if (lstr.good()) lstr>>cur_mat.Tr();
        } else if (token=="illum") {
            if (lstr.good()) lstr>>cur_mat.illum;
        } else if (token=="map_Ka") {
            std::string texname;
            do lstr>>texname; // drop all parameters
            while (!texname.empty() && texname[0]=='-');
            cur_mat.load_texture(MAP_KA,texname);
        } else if (token=="map_Kd") {
            std::string texname;
            do lstr>>texname; // drop all parameters
            while (!texname.empty() && texname[0]=='-');
            cur_mat.load_texture(MAP_KD,texname);
        } else if (token=="map_Ks") {
            std::string texname;
            do lstr>>texname; // drop all parameters
            while (!texname.empty() && texname[0]=='-');
            cur_mat.load_texture(MAP_KS,texname);
        } else if (token=="map_Ns") {
            std::string texname;
            do lstr>>texname; // drop all parameters
            while (!texname.empty() && texname[0]=='-');
            cur_mat.load_texture(MAP_NS,texname);
        } else if (token=="map_Tr" || token=="map_d") {
            std::string texname;
            while (lstr.good() && !lstr.str().empty()) lstr>>texname; // drop all parameters
            cur_mat.load_texture(MAP_TR,texname);
        } else if (token=="map_bump" || token=="bump") {
            std::string texname;
            while (lstr.good() && !lstr.str().empty()) lstr>>texname; // drop all parameters
            cur_mat.load_texture(MAP_BUMP,texname);
        } else if (token=="map_disp" || token=="disp") {
            std::string texname;
            while (lstr.good() && !lstr.str().empty()) lstr>>texname; // drop all parameters
            cur_mat.load_texture(MAP_DISP,texname);
        } else if (token=="map_decal" || token=="decal") {
            std::string texname;
            while (lstr.good() && !lstr.str().empty()) lstr>>texname; // drop all parameters
            cur_mat.load_texture(MAP_DECAL,texname);
        } else if (token.size()>0 && (token=="usemtl" || token[0]=='v' || token[0]=='f' || token[0]=='s' )) {
            file.seekg(lastline); // We just skipped past the end of a material definition
            break;
        }
        token.clear();
        lastline = file.tellg();
    }
    materials[matname]=cur_mat;
}

void Material::bind_gpu() {
    if (shader) {
        static glow::Uniform<glm::vec4> UKa, UKd, UKs;
        static glow::Uniform<GLfloat> UNs;
        static glow::Uniform<GLint> Uillum;
        UKa = Ka; shader->copy_uniform("Ka",UKa);
        UKd = Kd; shader->copy_uniform("Kd",UKd);
        UKs = Ks; shader->copy_uniform("Ks",UKs);
        UNs = Ns; shader->copy_uniform("Ns",UNs);
        Uillum = illum; shader->copy_uniform("illum",Uillum);
        static glow::Uniform<GLint> UTexTarget;
        static const std::vector<std::string> mapnames({"map_Ka", "map_Kd", "map_Ks", "map_Ns", "map_Tr", "map_bump", "map_disp", "map_decal"});
        for (int tm=0; tm<MAP_MAX; tm++) {
            if (_tex_tar[tm]) {
                glActiveTexture(_tex_tar[tm]);
                glBindTexture(GL_TEXTURE_2D,_texture[tm]);
                UTexTarget=_tex_tar[tm]-GL_TEXTURE0;
                shader->copy_uniform(mapnames[tm],UTexTarget);
            }
        }
        glActiveTexture(GL_TEXTURE0);
    }
}
void Material::bind_local() {
    static const glm::vec4 black(0.f,0.f,0.f,1.f);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    glBindTexture(GL_TEXTURE_2D,_texture[MAP_KD]);
    // Multiple textures require shaders or multipass rendering;
    // Both aren't practical here, so we just use the diffuse map.

    if (0 == illum) glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,&black[0]);
    else glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,&Ka[0]);
    glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,&Kd[0]);
    if (illum < 2) {
        glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,&black[0]);
        glMaterialf(GL_FRONT_AND_BACK,GL_SHININESS,0.f);
    } else {
        glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,&Ks[0]);
        glMaterialf(GL_FRONT_AND_BACK,GL_SHININESS,Ns);
    }

}

void Object::Submesh::draw() {
    if(mesh.on_gpu()) {
        Material::find(mtl).bind_gpu();
        mesh.draw_gpu();
    } else {
        Material::find(mtl).bind_local();
        mesh.draw_local();
    }
}

} // namespace glow
