#ifndef GLOW_WAVEFRONT_HPP
#define GLOW_WAVEFRONT_HPP

#include "glow.hpp"
#include <functional>
#include <string>
#include <sstream>

namespace glow {

struct ObjVertex {
    float x,y,z,w;
    float tu,tv,tw;
    float nx,ny,nz;
};

const VertexDataInfos OBJ_VF = {
    {0,GL_VERTEX_ARRAY,4,GL_FLOAT,GL_FALSE,sizeof(ObjVertex),(void*)offsetof(ObjVertex,x)},
    {1,GL_TEXTURE_COORD_ARRAY,3,GL_FLOAT,GL_FALSE,sizeof(ObjVertex),(void*)offsetof(ObjVertex,tu)},
    {2,GL_NORMAL_ARRAY,3,GL_FLOAT,GL_FALSE,sizeof(ObjVertex),(void*)offsetof(ObjVertex,nx)}
};

class Material;
typedef std::function<GLuint (const std::string &filename)> TextureLoader;
class Material {
public:
    typedef enum MAPS {MAP_KA, MAP_KD, MAP_KS, MAP_NS, MAP_TR, MAP_BUMP, MAP_DISP, MAP_DECAL, MAP_MAX} MapType;

    Material() {};

    void load_texture(MapType map_t, GLuint index) {_texture[map_t]=index;}
    void load_texture(MapType map_t, const std::string &filename) {load_texture(map_t,tex_loader(filename));}

    void bind_gpu();
    void bind_local();

    glm::vec4 Ka = glm::vec4(1.f,1.f,1.f,1.f);
    glm::vec4 Kd = glm::vec4(1.f,1.f,1.f,1.f);
    glm::vec4 Ks = glm::vec4(0.f,0.f,0.f,1.f);
    float Ns = 0.f;
    float &Tr() {return Kd.a;}
    float &d() {return Kd.a;}
    int illum = 1; // Modes 3-10 use raytracing, and are thus impossible in OGL

    static Shader *shader;
    static Material &find(const std::string &name) {return materials[name];}
    static void read(std::ifstream &file);
    static void load(const std::string &mtl_file) {
        std::ifstream fs(mtl_file, std::ios::in);
        if(fs.is_open()) {read(fs); fs.close();} }

    static TextureLoader tex_loader;
    static std::map<std::string,Material> materials;
    static GLenum &texture_target(MapType map_t) {return _tex_tar[map_t];}
private:
    GLuint _texture[MAP_MAX] = {0,0,0,0,0,0,0,0};
    static GLenum _tex_tar[MAP_MAX];
};

class Object {
    struct Submesh {
        Submesh(const std::string &material) : mesh(OBJ_VF), mtl(material) {};
        void draw();
        void upload() {if (!mesh.on_gpu()) mesh.make_buffers(); mesh.upload();}
        Geometry<ObjVertex,GLuint> mesh;
        std::string mtl;};
public:
    Object() {};
    Object(const std::string &path, bool flip_v = true) {load_mesh(path,flip_v);};

    void load_mesh(const std::string &path, bool flip_v = true);

    void upload() {for (auto &m : _meshes) m.upload();}
    void clear_local() {for (auto &m : _meshes) m.mesh.clear_local();}
    void clear_gpu() {for (auto &m : _meshes) m.mesh.clear_gpu();}
    void draw() {for (auto &m : _meshes) m.draw();}

private:
    std::vector<Submesh> _meshes;
};

} // namespace glow

#endif // GLOW_WAVEFRONT_HPP
