
#define SDW_OPENGL // Set this before including sdw/qdw if you want to use OpenGL
#include "qdw.hpp"
#include <iostream>

//#define SOFTWARE_GUI // Has no effect in sdw/qdw; only used in main() for easy testing.

void add_some_widgets(qdw::Gui &main_gui); // QDW demo

#ifdef SDW_OPENGL // To test if QDW and OGL can share a window
#define GLM_FORCE_RADIANS
#include <glm/glm.hpp> // I'm not going to do matrix operations by hand...
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtx/transform.hpp>
#include <fstream> // To load the shaders
struct GL_Handles {
    GLuint VertexArrayID,ProgramID,MatrixID;
    GLuint vertexbuffer,colorbuffer;
    glm::mat4 camera;
};
GL_Handles init_gl_stuff();
void draw_gl_stuff(GL_Handles &handles);
#endif

int main(int argc, char **argv)
try { // Yup, that's legal C++

    // Initialize some SDL stuff
    sdw::Context c(SDL_INIT_EVERYTHING,IMG_INIT_ALL,true);
#ifdef SDW_OPENGL // Using modern OGL in this example, just because we can.
    #ifndef SOFTWARE_GUI
    sdw::GL_Attributes gla;
    gla[SDL_GL_CONTEXT_MAJOR_VERSION]=3;
    gla[SDL_GL_CONTEXT_MINOR_VERSION]=3;
    gla[SDL_GL_CONTEXT_PROFILE_MASK]=SDL_GL_CONTEXT_PROFILE_COMPATIBILITY;
    gla[SDL_GL_MULTISAMPLEBUFFERS]=1;
    gla[SDL_GL_MULTISAMPLESAMPLES]=4;
    sdw::GL_Window win("SDL-Test",100,100,640,480,gla);
    #else
    sdw::Window win("SDL-Test",100,100,640,480);
    #endif
#else
    sdw::Window win("SDL-Test",100,100,640,480);
#endif

    // Something to show off SDW (and a renderer, if needed)
sdw::Font arial18("arial.ttf",18);
sdw::Font buxton64("buxton.ttf",64);
#ifdef SOFTWARE_GUI
    sdw::Surface img("image.png");
    sdw::Surface text = buxton64.write("Software Renderer");
#else
    #ifdef SDW_OPENGL
    sdw::Renderer ren(win,SDL_RENDERER_ACCELERATED); // I don't want vsync here (to test performance limits)
    sdw::Texture text = buxton64.write(ren,"OpenGL Renderer");
    GL_Handles glh = init_gl_stuff();
    #else
    sdw::Renderer ren(win,-1,SDL_RENDERER_ACCELERATED);
    sdw::Texture text = buxton64.write(ren,"Hardware Renderer");
    #endif
    sdw::Texture img(ren,"image.png");
#endif

    // And make a Gui with a bunch of widgets - try "tileset_windows.bmp" to see color masking
    qdw::Theme win_theme(sdw::Surface("tileset_windows.png"),sdw::Font("arial.ttf",12),16,2);
#ifdef SOFTWARE_GUI
    qdw::Gui main_gui(win_theme,win.rect());
#else
    qdw::TexGui main_gui(ren,win_theme,win.rect());
#endif
    add_some_widgets(main_gui); // For demo purposes; code is below main() for clarity
    // No need to worry about memory leaks/segfaults here; the Gui takes care of that (with a little help from std::shared_ptr<>)

    //Get the text w/h so we can center it in the screen
    int x = (win.width()-text.width()) / 2;
    int y = (win.height()-text.height()) / 2;

    int lastframe = SDL_GetTicks();
    int updated_frm = 0;
    int total_frm = 0;
    std::stringstream temp;

    //Our event structure
    SDL_Event e;
    bool quit = false;
    while (!quit) {
        while (SDL_PollEvent(&e)){
            if (!main_gui.hit(e)) { // Only process events the GUI didn't consume
                if (e.type == SDL_QUIT)
                    quit = true;
                if (e.type == SDL_KEYDOWN){
                    switch (e.key.keysym.sym){
                        case SDLK_ESCAPE:
                            quit = true;
                            break;
                        default:
                            break;
                    }
                }
            }
        }

        // Update Widgets (if necessary)
        if (main_gui.update()) updated_frm++;
        total_frm++;

        // Performance indicators
        int t = SDL_GetTicks();
        temp.str(""); temp.clear();
        temp<<t-lastframe<<" ms/frm";
        sdw::Surface tpf = arial18.write_solid(temp.str());
        temp.str(""); temp.clear();
        temp<<updated_frm<<"/"<<total_frm<<" upd";
        sdw::Surface upd = arial18.write_solid(temp.str());
        lastframe = t;

#ifdef SOFTWARE_GUI // Blit Gui directly (slower; fillrate on software renderer is usually poor)
        win.clear();
        img.blit((SDL_Surface*)win,sdw::Point(100,175));
        text.blit((SDL_Surface*)win,sdw::Point(x,y));
        main_gui.draw((SDL_Surface*)win);
        tpf.blit(win,sdw::Point(win.width()-tpf.width()-5,5));
        upd.blit(win,sdw::Point(win.width()-upd.width()-5,25));
        win.flip();
#else // Render with TexGui (very fast, especially if no Widget changed)
    #ifdef SDW_OPENGL // OpenGL renderer
        win.fill(0,0,0); // We don't need SDL's layers of indirection here. We know it's OpenGL, so we can use the
                         // GL_Window directly - we could even use glClear(GL_COLOR_BUFFER_BIT); straight away
        img.gl_render(sdw::Point(100,175));
        draw_gl_stuff(glh); // NB: Mixing GLSL with immediate mode here
        text.gl_render(sdw::Point(x,y));
        main_gui.texture.gl_render(sdw::Rect(0,0,win.width(),win.height()));
        sdw::Texture ttpf(ren,tpf); ttpf.gl_render(sdw::Point(win.width()-tpf.width()-5,5));
        sdw::Texture tupd(ren,upd); tupd.gl_render(sdw::Point(win.width()-upd.width()-5,25));
        win.flip();
    #else // Whatever hardware renderer SDL can get it's hands on
        ren.clear(); // Let SDL figure out how to do that
        img.render(sdw::Point(100,175));
        text.render(sdw::Point(x,y));
        main_gui.render();
        sdw::Texture ttpf(ren,tpf); ttpf.render(sdw::Point(win.width()-tpf.width()-5,5));
        sdw::Texture tupd(ren,upd); tupd.render(sdw::Point(win.width()-upd.width()-5,25));
        ren.flip();
    #endif
#endif

    }

    return 0;

} catch(sdw::Error e) {
    std::cerr<<"Error: "<<e<<std::endl;
    return 0;
} catch (...) {
    std::cerr<<"Unknown exception!"<<std::endl;
    return -1;
}

void add_some_widgets(qdw::Gui &main_gui) {
    main_gui.addLabel("lbl_title",sdw::Rect(100,290,250,80),"I'm a label!");
    sdw::Font buxton64("buxton.ttf",64);
    auto large_text = std::make_shared<qdw::Theme>(buxton64,sdw::Color(0,128,255));
    main_gui.get("lbl_title")->custom_theme = large_text;
    main_gui.addFrame("frm_large",sdw::Rect(10,10,450,200),0);
    main_gui.get("frm_large")->addScrollFrame("frm_ul",sdw::Rect(10,10,200,150),-2);

    main_gui.addSubWindow("win_test",sdw::Rect(100,100,320,200),"I'm a Window!");
    auto &swin = *main_gui.get<qdw::SubWindow>("win_test"); // We want a SubWindow&, not a Widget& here!
    swin.state|=qdw::Widget::ST_RESIZABLE;
    swin.set_button_states(true,true,false);

    main_gui.addSubWindow("win_two",sdw::Rect(250,250,225,225),"Yet another Window!");
    auto &swin2 = *main_gui.get<qdw::SubWindow>("win_two");
    swin2.set_button_states(false,false,true);

    auto &frm = *main_gui.get<qdw::ScrollFrame>("frm_ul");
    frm.state |= qdw::Widget::ST_RESIZABLE;
    frm.addCheckBox("chk1",sdw::Rect(25,100,120,20),"Check this!");
    frm.addCheckBox("chk2",sdw::Rect(25,120,120,20),"Or this!");
    frm.addCheckBox("chk3",sdw::Rect(25,140,120,20),"Or maybe this!");

    frm.addOptionButton("opt1a",sdw::Rect(150,100,120,20),"One option.");
    frm.addOptionButton("opt2a",sdw::Rect(150,120,120,20),"Another option.");
    frm.addOptionButton("opt3a",sdw::Rect(150,140,120,20),"So many options.");

    frm.addOptionButton("opt1b",sdw::Rect(150,200,120,20),"Different block.",1);
    frm.addOptionButton("opt2b",sdw::Rect(150,220,120,20),"With more options.",1);
    frm.addOptionButton("opt3b",sdw::Rect(150,240,120,20),"Three, exactly.",1);

    frm.addEditBox("ed1",sdw::Rect(25,180,120,24));
    auto &ed1 = *main_gui.get<qdw::EditBox>("ed1");
    ed1.text="Edit me!";
    ed1.textEdit = [](qdw::Widget*w,const SDL_Event&){std::cout<<"My text is now: "<<static_cast<qdw::EditBox*>(w)->text<<std::endl;};

    swin.addTextBox("txt_test",sdw::Rect(0,0,260,175));
    auto &txtbox = *main_gui.get<qdw::TextBox>("txt_test");
    txtbox.bars = qdw::ScrollFrame::HBAR_AUTO|qdw::ScrollFrame::VBAR_AUTO|qdw::ScrollFrame::RBTN_AUTO;
    txtbox.state|=qdw::Widget::ST_RESIZABLE;
    txtbox.text = "Lorem ipsum dolor sit amet, consectetur adipisicing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.";
    auto tbox_theme = std::make_shared<qdw::Theme>(sdw::Font("buxton.ttf",14),sdw::Color(0,0,128));
    txtbox.custom_theme = tbox_theme;
    //txtbox.readonly=true;
    //txtbox.wraplines=false;

    swin2.addGrid("grid1",sdw::Rect(0,0,220,200),32,32);
    auto &grid1 = *main_gui.get<qdw::Grid>("grid1");
    grid1.cell.x=grid1.cell.y=8;
    grid1.adjust=qdw::Grid::AJ_CENTER;

    swin2.addIcon("ico_test",sdw::Rect(100,100,32,32),sdw::Surface("icon.png"),"Drag me!");
    auto &ico = *main_gui.get<qdw::Icon>("ico_test");
    ico.make_surfaces();
    ico.caption_pos=qdw::Icon::CP_UNDER;
    ico.mouseDragStart = [](qdw::Widget*,const SDL_Event&){std::cout<<"Hey! Put me down!\n";};
    ico.mouseDragStop = [](qdw::Widget*,const SDL_Event&){std::cout<<"Thanks!\n";};
    ico.mouseDrop = [&](qdw::Widget *a,qdw::Widget *p,const SDL_Event&){std::cout<<"I was dropped on '"<<main_gui.get_id(p)<<"'"<<std::endl;};
    ico.scaleable = true;

    frm.addButton("btn1",sdw::Rect(25,25,120,32),"Click me!",
                     [](qdw::Widget*,const SDL_Event&)
                     {std::cout<<"I was clicked."<<std::endl;} );
    frm.addButton("btn2",sdw::Rect(25,60,120,32),"I'm sticky!",
                     [&](qdw::Widget* w,const SDL_Event&)
                     {std::cout<<"Button state: "<<w->checked()<<std::endl; if (w->checked()) swin.disable(); else swin.enable();} );

    frm.addButton("btn3",sdw::Rect(25,225,120,32),"Find me!",
                     [](qdw::Widget*,const SDL_Event&)
                     {std::cout<<"You got me!"<<std::endl;} );

    auto &btn1 = *main_gui.get<qdw::Button>("btn1");
    auto &btn2 = *main_gui.get<qdw::Button>("btn2");

    btn2.sticky = true;

    auto button_theme = std::make_shared<qdw::Theme>(sdw::Font("arial.ttf",18),sdw::Color(128,0,0));
    btn1.custom_theme = button_theme;
    btn1.setHotkey('a',KMOD_CTRL,[](qdw::Widget*,const SDL_Event&){std::cout<<"Hotkey CTRL+A"<<std::endl;} );
    btn2.setHotkey('b',KMOD_SHIFT,[](qdw::Widget*,const SDL_Event&){std::cout<<"Hotkey SHIFT+B"<<std::endl;} );
}

#ifdef SDW_OPENGL // To test if QDW and OGL can share a window
GLuint LoadShaders(const char * vertex_file_path, const char * fragment_file_path);
GL_Handles init_gl_stuff() {
    GL_Handles handles;

    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LESS);
    glEnable(GL_CULL_FACE);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    glGenVertexArrays(1,&handles.VertexArrayID);
    glBindVertexArray(handles.VertexArrayID);

    static const GLfloat g_vertex_buffer_data[] = {
        -1.0f,-1.0f,-1.0f, // triangle 1 : begin
        -1.0f,-1.0f, 1.0f,
        -1.0f, 1.0f, 1.0f, // triangle 1 : end
        1.0f, 1.0f,-1.0f, // triangle 2 : begin
        -1.0f,-1.0f,-1.0f,
        -1.0f, 1.0f,-1.0f, // triangle 2 : end
        1.0f,-1.0f, 1.0f,
        -1.0f,-1.0f,-1.0f,
        1.0f,-1.0f,-1.0f,
        1.0f, 1.0f,-1.0f,
        1.0f,-1.0f,-1.0f,
        -1.0f,-1.0f,-1.0f,
        -1.0f,-1.0f,-1.0f,
        -1.0f, 1.0f, 1.0f,
        -1.0f, 1.0f,-1.0f,
        1.0f,-1.0f, 1.0f,
        -1.0f,-1.0f, 1.0f,
        -1.0f,-1.0f,-1.0f,
        -1.0f, 1.0f, 1.0f,
        -1.0f,-1.0f, 1.0f,
        1.0f,-1.0f, 1.0f,
        1.0f, 1.0f, 1.0f,
        1.0f,-1.0f,-1.0f,
        1.0f, 1.0f,-1.0f,
        1.0f,-1.0f,-1.0f,
        1.0f, 1.0f, 1.0f,
        1.0f,-1.0f, 1.0f,
        1.0f, 1.0f, 1.0f,
        1.0f, 1.0f,-1.0f,
        -1.0f, 1.0f,-1.0f,
        1.0f, 1.0f, 1.0f,
        -1.0f, 1.0f,-1.0f,
        -1.0f, 1.0f, 1.0f,
        1.0f, 1.0f, 1.0f,
        -1.0f, 1.0f, 1.0f,
        1.0f,-1.0f, 1.0f
    };

    static const GLfloat g_color_buffer_data[] = {
        0.583f,  0.771f,  0.014f, 1.f,
        0.609f,  0.115f,  0.436f, 1.f,
        0.327f,  0.483f,  0.844f, 1.f,
        0.822f,  0.569f,  0.201f, 1.f,
        0.435f,  0.602f,  0.223f, 1.f,
        0.310f,  0.747f,  0.185f, 1.f,
        0.597f,  0.770f,  0.761f, 1.f,
        0.559f,  0.436f,  0.730f, 1.f,
        0.359f,  0.583f,  0.152f, 1.f,
        0.483f,  0.596f,  0.789f, 1.f,
        0.559f,  0.861f,  0.639f, 1.f,
        0.195f,  0.548f,  0.859f, 1.f,
        0.014f,  0.184f,  0.576f, 1.f,
        0.771f,  0.328f,  0.970f, 1.f,
        0.406f,  0.615f,  0.116f, 1.f,
        0.676f,  0.977f,  0.133f, 1.f,
        0.971f,  0.572f,  0.833f, 1.f,
        0.140f,  0.616f,  0.489f, 1.f,
        0.997f,  0.513f,  0.064f, 1.f,
        0.945f,  0.719f,  0.592f, 1.f,
        0.543f,  0.021f,  0.978f, 1.f,
        0.279f,  0.317f,  0.505f, 1.f,
        0.167f,  0.620f,  0.077f, 1.f,
        0.347f,  0.857f,  0.137f, 1.f,
        0.055f,  0.953f,  0.042f, 1.f,
        0.714f,  0.505f,  0.345f, 1.f,
        0.783f,  0.290f,  0.734f, 1.f,
        0.722f,  0.645f,  0.174f, 1.f,
        0.302f,  0.455f,  0.848f, 1.f,
        0.225f,  0.587f,  0.040f, 1.f,
        0.517f,  0.713f,  0.338f, 1.f,
        0.053f,  0.959f,  0.120f, 1.f,
        0.393f,  0.621f,  0.362f, 1.f,
        0.673f,  0.211f,  0.457f, 1.f,
        0.820f,  0.883f,  0.371f, 1.f,
        0.982f,  0.099f,  0.879f, 1.f
    };

    glGenBuffers(1, &handles.vertexbuffer);
    glBindBuffer(GL_ARRAY_BUFFER, handles.vertexbuffer);
    glBufferData(GL_ARRAY_BUFFER, sizeof(g_vertex_buffer_data), g_vertex_buffer_data, GL_STATIC_DRAW);

    glGenBuffers(1, &handles.colorbuffer);
    glBindBuffer(GL_ARRAY_BUFFER, handles.colorbuffer);
    glBufferData(GL_ARRAY_BUFFER, sizeof(g_color_buffer_data), g_color_buffer_data, GL_STATIC_DRAW);

    handles.ProgramID = LoadShaders("simple.vs","simple.fs");

    glm::mat4 Projection = glm::perspective((float)M_PI/4.f, 4.0f / 3.0f, 0.1f, 100.0f);
    glm::mat4 View       = glm::lookAt(glm::vec3(2,6,2),glm::vec3(0,0,1),glm::vec3(0,0,1));
    glm::mat4 Model      = glm::mat4(1.0f);
    handles.camera       = Projection * View * Model;

    handles.MatrixID = glGetUniformLocation(handles.ProgramID, "MVP");

    return handles;
}
void draw_gl_stuff(GL_Handles &handles) {
        glClear(GL_DEPTH_BUFFER_BIT);

        glEnableVertexAttribArray(0);
        glBindBuffer(GL_ARRAY_BUFFER, handles.vertexbuffer);
        glVertexAttribPointer(0,3,GL_FLOAT,GL_FALSE,0,NULL);

        glEnableVertexAttribArray(1);
        glBindBuffer(GL_ARRAY_BUFFER, handles.colorbuffer);
        glVertexAttribPointer(1,4,GL_FLOAT,GL_FALSE,0,NULL);

        glUseProgram(handles.ProgramID);

        glUniformMatrix4fv(handles.MatrixID, 1, GL_FALSE, &handles.camera[0][0]);

        glDrawArrays(GL_TRIANGLES, 0, 12*3);

        glDisableVertexAttribArray(0);
        glDisableVertexAttribArray(1);
        glUseProgram(0);
        glDisable(GL_DEPTH_TEST);
        glDisable(GL_CULL_FACE);
}
GLuint LoadShaders(const char *vertex_file_path,const char *fragment_file_path) {

    // Create the shaders
    GLuint VertexShaderID = glCreateShader(GL_VERTEX_SHADER);
    GLuint FragmentShaderID = glCreateShader(GL_FRAGMENT_SHADER);

    // Read the Vertex Shader code from the file
    std::string VertexShaderCode;
    std::ifstream VertexShaderStream(vertex_file_path, std::ios::in);
    if(VertexShaderStream.is_open())
    {
        std::string Line = "";
        while(getline(VertexShaderStream, Line))
            VertexShaderCode += "\n" + Line;
        VertexShaderStream.close();
    }

    // Read the Fragment Shader code from the file
    std::string FragmentShaderCode;
    std::ifstream FragmentShaderStream(fragment_file_path, std::ios::in);
    if(FragmentShaderStream.is_open()){
        std::string Line = "";
        while(getline(FragmentShaderStream, Line))
            FragmentShaderCode += "\n" + Line;
        FragmentShaderStream.close();
    }

    GLint Result = GL_FALSE;
    int InfoLogLength;

    // Compile Vertex Shader
    printf("Compiling shader : %s\n", vertex_file_path);
    char const * VertexSourcePointer = VertexShaderCode.c_str();
    glShaderSource(VertexShaderID, 1, &VertexSourcePointer , NULL);
    glCompileShader(VertexShaderID);

    // Check Vertex Shader
    glGetShaderiv(VertexShaderID, GL_COMPILE_STATUS, &Result);
    glGetShaderiv(VertexShaderID, GL_INFO_LOG_LENGTH, &InfoLogLength);
    std::vector<char> VertexShaderErrorMessage(InfoLogLength);
    glGetShaderInfoLog(VertexShaderID, InfoLogLength, NULL, &VertexShaderErrorMessage[0]);
    fprintf(stdout, "%s\n", &VertexShaderErrorMessage[0]);

    // Compile Fragment Shader
    printf("Compiling shader : %s\n", fragment_file_path);
    char const * FragmentSourcePointer = FragmentShaderCode.c_str();
    glShaderSource(FragmentShaderID, 1, &FragmentSourcePointer , NULL);
    glCompileShader(FragmentShaderID);

    // Check Fragment Shader
    glGetShaderiv(FragmentShaderID, GL_COMPILE_STATUS, &Result);
    glGetShaderiv(FragmentShaderID, GL_INFO_LOG_LENGTH, &InfoLogLength);
    std::vector<char> FragmentShaderErrorMessage(InfoLogLength);
    glGetShaderInfoLog(FragmentShaderID, InfoLogLength, NULL, &FragmentShaderErrorMessage[0]);
    fprintf(stdout, "%s\n", &FragmentShaderErrorMessage[0]);

    // Link the program
    fprintf(stdout, "Linking program\n");
    GLuint ProgramID = glCreateProgram();
    glAttachShader(ProgramID, VertexShaderID);
    glAttachShader(ProgramID, FragmentShaderID);
    glLinkProgram(ProgramID);

    // Check the program
    glGetProgramiv(ProgramID, GL_LINK_STATUS, &Result);
    glGetProgramiv(ProgramID, GL_INFO_LOG_LENGTH, &InfoLogLength);
    std::vector<char> ProgramErrorMessage( std::max(InfoLogLength, int(1)) );
    glGetProgramInfoLog(ProgramID, InfoLogLength, NULL, &ProgramErrorMessage[0]);
    fprintf(stdout, "%s\n", &ProgramErrorMessage[0]);

    glDeleteShader(VertexShaderID);
    glDeleteShader(FragmentShaderID);

    return ProgramID;
}
#endif
