#ifndef SDW_HPP
#define SDW_HPP

#ifdef SDW_OPENGL
#include <GL/glew.h>
#include <cstdlib>
#endif

#ifdef __linux__
#include <SDL2/SDL.h>
#include <SDL2/SDL_image.h>
#include <SDL2/SDL_ttf.h>
#ifdef SDW_OPENGL
#include <SDL2/SDL_opengl.h>
#endif
#elif _WIN32
#include <SDL.h>
#include <SDL_image.h>
#include <SDL_ttf.h>
#ifdef SDW_OPENGL
#include <SDL_opengl.h>
#endif
#endif

#include <string>
#include <vector>
#include <memory>
#include <cmath>
#include <map>

#include <iostream>

const int IMG_INIT_ALL = IMG_INIT_JPG | IMG_INIT_PNG | IMG_INIT_TIF | IMG_INIT_WEBP;

namespace sdw {

    // ITU-R Recommendation BT.709
    const float R_LUM = .2126f;
    const float G_LUM = .7152f;
    const float B_LUM = .0722f;


#if SDL_BYTEORDER == SDL_BIG_ENDIAN
    const Uint32 RMASK = 0xff000000;
    const Uint32 GMASK = 0x00ff0000;
    const Uint32 BMASK = 0x0000ff00;
    const Uint32 AMASK = 0x000000ff;

    inline Uint32 MapColor(SDL_Color c) {return (Uint32)c.r<<24 | (Uint32)c.g<<16 | (Uint32)c.b<<8 | c.a;}
    inline Uint32 MapRGB(Uint8 r, Uint8 g, Uint8 b) {return (Uint32)r<<24 | (Uint32)g<<16 | (Uint32)b<<8 | 255;}
    inline Uint32 MapRGBA(Uint8 r, Uint8 g, Uint8 b, Uint8 a) {return (Uint32)r<<24 | (Uint32)g<<16 | (Uint32)b<<8 | a;}
    inline SDL_Color UnmapColor(Uint32 c) {return {Uint8(c>>24),Uint8((c&GMASK)>>16),Uint8((c&BMASK)>>8),Uint8(c&AMASK)};}
#else
    const Uint32 RMASK = 0x000000ff;
    const Uint32 GMASK = 0x0000ff00;
    const Uint32 BMASK = 0x00ff0000;
    const Uint32 AMASK = 0xff000000;

    inline Uint32 MapColor(SDL_Color c) {return c.r | (Uint32)c.g<<8 | (Uint32)c.b<<16 | (Uint32)c.a<<24;}
    inline Uint32 MapRGB(Uint8 r, Uint8 g, Uint8 b) {return (Uint32)r | (Uint32)g<<8 | (Uint32)b<<16 | (255<<24);}
    inline Uint32 MapRGBA(Uint8 r, Uint8 g, Uint8 b, Uint8 a) {return r | (Uint32)g<<8 | (Uint32)b<<16 | (Uint32)a<<24;}
    inline SDL_Color UnmapColor(Uint32 c) {return {Uint8(c&RMASK),Uint8((c&GMASK)>>8),Uint8((c&BMASK)>>16),Uint8(c>>24)};}
#endif

typedef std::string Error; // SDL Errors are char*
// But throwing char* around is bound to turn exceptions into segfaults.

inline void Assert(bool b) throw(Error) {if (!b) throw(Error(SDL_GetError()));}

// A bunch of pseudo-constructors
inline SDL_Point Point(int x, int y) {return {x,y};}
inline SDL_Rect Rect(int x, int y, int w, int h) {return {x,y,w,h};}
inline SDL_Color Color(Uint8 r, Uint8 g, Uint8 b, Uint8 a = 255) {return {r,g,b,a};}

// Some helper functions
inline Uint16 permissive(Uint16 mods) { //  SDL_Keymod is Uint16
    if (mods&KMOD_SHIFT) mods|=KMOD_SHIFT;
    if (mods&KMOD_CTRL) mods|=KMOD_CTRL;
    if (mods&KMOD_ALT) mods|=KMOD_ALT;
    if (mods&KMOD_GUI) mods|=KMOD_GUI;
    return mods;
}
inline bool similar(Uint16 m1, Uint16 m2) { //  SDL_Keymod is Uint16
    return permissive(m1)==permissive(m2);
}
inline SDL_Color operator*(const SDL_Color& c1, const SDL_Color& c2) {
    SDL_Color c;
    c.r = ((int)c1.r*c2.r)>>8;
    c.g = ((int)c1.g*c2.g)>>8;
    c.b = ((int)c1.b*c2.b)>>8;
    c.a = ((int)c1.a*c2.a)>>8;
    return c;
}
inline SDL_Color mix(const SDL_Color& c1, const SDL_Color& c2, float p1 = .5f) {
    SDL_Color c;
    c.r = (c1.r*p1)+(c2.r*(1.f-p1));
    c.g = (c1.g*p1)+(c2.g*(1.f-p1));
    c.b = (c1.b*p1)+(c2.b*(1.f-p1));
    c.a = (c1.a*p1)+(c2.a*(1.f-p1));
    return c;
}
inline float luminance(const SDL_Color& c) {return R_LUM*c.r + G_LUM*c.g + B_LUM*c.b;}

// And some very basic geometry
inline float Distance(SDL_Point p1, SDL_Point p2) {return hypotf(p1.x-p2.x,p1.y-p2.x);}
inline bool Inside(SDL_Point p, SDL_Rect r) {
    return (p.x>=r.x)&&(p.y>=r.y)&&(p.x<=r.x+r.w)&&(p.y<=r.y+r.h);}
inline bool Inside(const SDL_Event &e, const SDL_Rect &r) {
    if ((e.type == SDL_MOUSEBUTTONDOWN) || (e.type == SDL_MOUSEBUTTONUP)) return Inside(Point(e.button.x,e.button.y),r);
    else if ((e.type == SDL_MOUSEMOTION)) return Inside(Point(e.motion.x,e.motion.y),r);
    else return false;
}
inline bool Inside(SDL_Rect inner, SDL_Rect outer) {
    return (inner.x>=outer.x)&&(inner.y>=outer.y)&&(inner.x+inner.w<=outer.x+outer.w)&&(inner.y+inner.h<=outer.y+outer.h);}
inline bool Intersect(SDL_Rect r1, SDL_Rect r2) {
    return ((r1.x+r1.w>r2.x)||(r2.x+r2.w>r1.x))&&((r1.y+r1.h>r2.y)||(r2.y+r2.h>r1.y));}
inline SDL_Rect operator| (const SDL_Rect &r1, const SDL_Rect &r2) {
    SDL_Rect r=r1;
    if (r2.x<r1.x) r.x=r2.x;
    if (r2.x+r2.w>r1.x+r1.w) r.w=r2.x+r2.w-r.x;
    else r.w=r1.x+r1.w-r.x;
    if (r2.y<r1.y) r.y=r2.y;
    if (r2.y+r2.h>r1.y+r1.h) r.h=r2.y+r2.h-r.y;
    else r.h=r1.y+r1.h-r.y;
    return r;
}
inline SDL_Rect operator& (const SDL_Rect &r1, const SDL_Rect &r2) {
    SDL_Rect r=r1;
    if (r2.x>r1.x) r.x=r2.x;
    if (r2.x+r2.w<r1.x+r1.w) r.w=r2.x+r2.w-r.x;
    else r.w=r1.x+r1.w-r.x;
    if (r.w<0) r.w=0;
    if (r2.y>r1.y) r.y=r2.y;
    if (r2.y+r2.h<r1.y+r1.h) r.h=r2.y+r2.h-r.y;
    else r.h=r1.y+r1.h-r.y;
    if (r.h<0) r.h=0;
    return r;
}
inline SDL_Rect operator+ (const SDL_Rect &r, const SDL_Point &p) {
    SDL_Rect out = r;
    out.x+=p.x;
    out.y+=p.y;
    return out;
}
inline SDL_Rect operator+ (const SDL_Point &p, const SDL_Rect &r) {return r+p;}
inline SDL_Rect operator- (const SDL_Rect &r, const SDL_Point &p) {
    SDL_Rect out = r;
    out.x-=p.x;
    out.y-=p.y;
    return out;
}
inline SDL_Rect operator- (const SDL_Point &p, const SDL_Rect &r) {return r-p;}
inline SDL_Event operator+ (const SDL_Event &e, const SDL_Point &p) {
    SDL_Event ev = e;
    if ((e.type == SDL_MOUSEBUTTONDOWN) || (e.type == SDL_MOUSEBUTTONUP)) {
        ev.button.x+=p.x;
        ev.button.y+=p.y;
    }
    if ((e.type == SDL_MOUSEMOTION)) {
        ev.motion.x+=p.x;
        ev.motion.y+=p.y;
    }
    return ev;
}
inline SDL_Event operator+ (const SDL_Point &p, const SDL_Event &e) {return e+p;}
inline SDL_Event operator- (const SDL_Event &e, const SDL_Point &p) {
    SDL_Event ev = e;
    if ((e.type == SDL_MOUSEBUTTONDOWN) || (e.type == SDL_MOUSEBUTTONUP)) {
        ev.button.x-=p.x;
        ev.button.y-=p.y;
    }
    if ((e.type == SDL_MOUSEMOTION)) {
        ev.motion.x-=p.x;
        ev.motion.y-=p.y;
    }
    return ev;
}
inline SDL_Event operator- (const SDL_Point &p, const SDL_Event &e) {return e-p;}

class Context {
public:
    Context(int sdl_modules = SDL_INIT_EVERYTHING) throw(Error) {

        Assert(0==SDL_Init(sdl_modules));
        //init_glew();
    }
    Context(int sdl_modules, int image_formats, bool fonts) throw(Error) {

        Assert(0==SDL_Init(sdl_modules));
        if (image_formats) _images=IMG_Init(image_formats);
        if (fonts && 0==TTF_Init()) _fonts=true;
        //init_glew();
    }
    ~Context() {
        if (_images) IMG_Quit();
        SDL_Quit();
    };

    int images_available() {return _images;}
    bool images_available(int types) {return _images&types;}
    bool fonts_available() {return _fonts;}

protected:
    #ifdef SDW_OPENGL
    void init_glew() throw(Error) {
        glewExperimental = true;
        if (glewInit() != GLEW_OK) throw Error("Failed to initialize GLEW!");
        return;
    }
    #else
    void init_glew() throw(Error) {} // We don't need it
    #endif

    int _images=0;
    bool _fonts=false;
};

class Window {
public:
    Window() {};
    Window(const std::string &title, int x, int y, int w, int h, Uint32 flags = SDL_WINDOW_SHOWN) throw(Error) :
            _window(SDL_CreateWindow(title.c_str(), x, y, w, h, flags),SDL_DestroyWindow) {Assert((bool)_window);}

    operator SDL_Window*() const {return &*_window;}
    operator SDL_Surface*() const {return SDL_GetWindowSurface(&*_window);}

    SDL_Rect rect() const {
        int w,h;
        SDL_GetWindowSize(&*_window,&w,&h);
        return {0,0,w,h};
    }
    int width() const {
        int w;
        SDL_GetWindowSize(&*_window,&w,NULL);
        return w;
    }
    int height() const {
        int h;
        SDL_GetWindowSize(&*_window,NULL,&h);
        return h;
    }

    void clear() {
        SDL_Surface *ws = SDL_GetWindowSurface(&*_window);
        SDL_FillRect(ws,nullptr,SDL_MapRGB(ws->format,0,0,0));
    }
    void fill(Uint8 r, Uint8 g, Uint8 b, Uint8 a = 255) {
        SDL_Surface *ws = SDL_GetWindowSurface(&*_window);
        SDL_FillRect(ws,nullptr,SDL_MapRGBA(ws->format,r,g,b,a));
    }
    void flip() {SDL_UpdateWindowSurface(&*_window);}
    void update() {SDL_UpdateWindowSurface(&*_window);}

    void draw(SDL_Surface *surf, SDL_Rect src, SDL_Rect dst) const {
        SDL_BlitSurface(surf,&src,*this,&dst);
    }
    void draw(SDL_Surface *surf, SDL_Rect dst) const {
        SDL_BlitSurface(surf,nullptr,*this,&dst);
    }
    void draw(SDL_Surface *surf, SDL_Rect src, SDL_Point pos) const {
        SDL_Rect dst = {pos.x,pos.y,width(),height()};
        SDL_BlitSurface(surf,&src,*this,&dst);
    }
    void draw(SDL_Surface *surf, SDL_Point pos) const {
        SDL_Rect dst = {pos.x,pos.y,width(),height()};
        SDL_BlitSurface(surf,nullptr,*this,&dst);
    }

protected:
    std::shared_ptr<SDL_Window> _window;
};

#ifdef SDW_OPENGL
typedef std::map<SDL_GLattr,int> GL_Attributes;
class GL_Window {
public:
    GL_Window() {};
    GL_Window(const std::string &title, int x, int y, int w, int h, Uint32 flags = SDL_WINDOW_SHOWN) throw(Error) :
            _window(SDL_CreateWindow(title.c_str(), x, y, w, h, flags|SDL_WINDOW_OPENGL),SDL_DestroyWindow)
            {Assert((bool)_window); _context = SDL_GL_CreateContext(&*_window); Assert((bool)_context); init_glew();}
    GL_Window(const std::string &title, int x, int y, int w, int h, const GL_Attributes &gl_attr,
            Uint32 flags = SDL_WINDOW_SHOWN) throw(Error) {
            for (auto &a : gl_attr) SDL_GL_SetAttribute(a.first, a.second);
            _window = std::shared_ptr<SDL_Window>(SDL_CreateWindow(title.c_str(), x, y, w, h, flags|SDL_WINDOW_OPENGL),SDL_DestroyWindow);
            Assert((bool)_window);
            _context = SDL_GL_CreateContext(&*_window);
            Assert((bool)_context);
            init_glew();
        }
    ~GL_Window() {
        #ifndef __linux__
        SDL_GL_DeleteContext(_context); // Crashes in X11
        #endif
        }

    operator SDL_Window*() const {return &*_window;}

    SDL_Rect rect() const {
        int w,h;
        SDL_GetWindowSize(&*_window,&w,&h);
        return {0,0,w,h};
    }
    int width() const {
        int w;
        SDL_GetWindowSize(&*_window,&w,NULL);
        return w;
    }
    int height() const {
        int h;
        SDL_GetWindowSize(&*_window,NULL,&h);
        return h;
    }

    void clear(GLbitfield mask = GL_COLOR_BUFFER_BIT) {
        glClearColor(0.0,0.0,0.0,1.0);
        glClear(mask);
    }
    void fill(Uint8 r, Uint8 g, Uint8 b, Uint8 a = 255) {
        glClearColor(r/255.f,g/255.f,b/255.f,a/255.f);
        glClear(GL_COLOR_BUFFER_BIT);
    }
    void flip() {SDL_GL_SwapWindow(&*_window);}
    void swap() {SDL_GL_SwapWindow(&*_window);}

    int gl_renderer() {
        int ogl_ren = -1;
        int nRD = SDL_GetNumRenderDrivers();
        for(int i=0; i<nRD; i++) {
            SDL_RendererInfo info;
            if(!SDL_GetRenderDriverInfo(i, &info) && !strcmp(info.name, "opengl"))
                ogl_ren = i;
        }
        return ogl_ren;
    }

protected:
    void init_glew() throw(Error) {
        glewExperimental = true;
        if (glewInit() != GLEW_OK) throw Error("Failed to initialize GLEW!");
        return;
    }

    SDL_GLContext _context;
    std::shared_ptr<SDL_Window> _window;
};

#endif

class Renderer {
public:
    Renderer() : _window(nullptr) {};
#ifdef SDW_OPENGL
    Renderer(GL_Window &window, Uint32 flags = SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC) throw(Error) :
            Renderer(window,window.gl_renderer(),flags) {};
    Renderer(Window &window, int index = -1, Uint32 flags = SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC) throw(Error) :
            Renderer(static_cast<SDL_Window*>(window),index,flags) {};
    Renderer(SDL_Window *window, int index, Uint32 flags) throw(Error) :
            _renderer(SDL_CreateRenderer(window, index, flags),SDL_DestroyRenderer), _window(window) {
        Assert((bool)_renderer); if (flags&SDL_RENDERER_PRESENTVSYNC) SDL_GL_SetSwapInterval(1);}
#else
    Renderer(SDL_Window *window, int index = -1, Uint32 flags = SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC) throw(Error) :
            _renderer(SDL_CreateRenderer(window, index, flags),SDL_DestroyRenderer), _window(window) {Assert((bool)_renderer);}
#endif

    operator SDL_Renderer*() const {return &*_renderer;}
    operator SDL_Window*() const {return _window;}

    void clear() {SDL_RenderClear(&*_renderer);}
    void flip() {SDL_RenderPresent(&*_renderer);}
    void present() {SDL_RenderPresent(&*_renderer);}

protected:
    std::shared_ptr<SDL_Renderer> _renderer;
    SDL_Window *_window;
};

class Surface {
public:
    Surface() : _surface(nullptr) {};
    Surface(SDL_Surface *surf) throw(Error) : _surface(surf,SDL_FreeSurface) {Assert((bool)_surface);}
    Surface(int w, int h, Uint32 flags = SDL_SWSURFACE) throw(Error) :
        _surface(SDL_CreateRGBSurface(flags,w,h,32,RMASK,GMASK,BMASK,AMASK),SDL_FreeSurface) {Assert((bool)_surface);}
    Surface(const char *file) throw(Error) : _surface(IMG_Load(file),SDL_FreeSurface) {Assert((bool)_surface);}
    Surface(const std::string &file) throw(Error) : _surface(IMG_Load(file.c_str()),SDL_FreeSurface) {Assert((bool)_surface);}

    SDL_Rect rect() const {
        if ((bool)_surface) return {0,0,_surface->w,_surface->h};
        else return {0,0,0,0};}
    int width() const {
        if ((bool)_surface) return _surface->w;
        else return 0;}
    int height() const {
        if ((bool)_surface) return _surface->h;
        else return 0;}

    Uint32 &pixel(unsigned x, unsigned y) throw(Error) {
        if ((int)x>=width() || (int)y>=height()) throw(Error("Pixel coordinates outside surface."));
        Uint32* pixels = (Uint32*)_surface->pixels;
        return pixels[(y*(_surface->w))+x];
    }
    SDL_Color get_pixel(unsigned x, unsigned y) {
        SDL_Color out;
        SDL_GetRGBA(pixel(x,y),_surface->format,&out.r,&out.g,&out.b,&out.a);
        return out;
    }
    void set_pixel(unsigned x, unsigned y, SDL_Color c) {
        Uint32 ccode = mapRGBA(c);
        pixel(x,y)=ccode;
    }

    void clear() {
        SDL_FillRect(&*_surface,nullptr,0);
    }
    void fill(SDL_Color c) {
        SDL_FillRect(&*_surface,nullptr,mapRGBA(c));
    }
    void fill_rect(const SDL_Rect &dst, SDL_Color c) {
        SDL_FillRect(&*_surface,&dst,mapRGBA(c));
    }

    void setBlending(SDL_BlendMode bm) {SDL_SetSurfaceBlendMode(&*_surface,bm);}
    SDL_BlendMode getBlending() const {SDL_BlendMode bm; SDL_GetSurfaceBlendMode(&*_surface,&bm); return bm;}

    void setTransparency(Uint8 r, Uint8 g, Uint8 b) {SDL_SetColorKey(&*_surface,SDL_TRUE,SDL_MapRGB(_surface->format,r,g,b));}
    void setTransparency(Uint32 colorcode) {SDL_SetColorKey(&*_surface,SDL_TRUE,colorcode);}
    void unsetTransparency() {SDL_SetColorKey(&*_surface,SDL_FALSE,transparency());}
    Uint32 transparency() {
        Uint32 key;
        if (SDL_GetColorKey(&*_surface,&key)) return 0;
        else return key;
    }
    void replaceColor(SDL_Color replace, SDL_Color with) {
        replaceColor(mapRGBA(replace),mapRGBA(with));
    }
    void replaceColor(Uint32 replace, Uint32 with) {
        int sz=width()*height();
        if (!sz) return;
        Uint32* pixels = (Uint32*)_surface->pixels;
        for (int i = 0; i<sz; i++) if (pixels[i]==replace) pixels[i]=with;
    }

    Uint32 mapRGB(SDL_Color c) {return SDL_MapRGB(_surface->format,c.r,c.g,c.b);}
    Uint32 mapRGBA(SDL_Color c) {return SDL_MapRGBA(_surface->format,c.r,c.g,c.b,c.a);}
    Uint32 mapRGB(Uint8 r, Uint8 g, Uint8 b ) {return SDL_MapRGB(_surface->format,r,g,b);}
    Uint32 mapRGBA(Uint8 r, Uint8 g, Uint8 b, Uint8 a) {return SDL_MapRGBA(_surface->format,r,g,b,a);}
    SDL_Color unmap(Uint32 pixel) {
        SDL_Color out;
        SDL_GetRGBA(pixel,_surface->format,&out.r,&out.g,&out.b,&out.a);
        return out;
    }

    operator SDL_Surface*() const {return &*_surface;}

    void blit(SDL_Surface *target, SDL_Rect src, SDL_Rect dst) const {
        if (_surface) SDL_BlitSurface(&*_surface,&src,target,&dst);
    }
    void blit(SDL_Surface *target, SDL_Rect dst) const {
        if (_surface) SDL_BlitSurface(&*_surface,nullptr,target,&dst);
    }
    void blit(SDL_Surface *target, SDL_Rect src, SDL_Point pos) const {
        if (_surface) {
            SDL_Rect dst = {pos.x,pos.y,width(),height()};
            SDL_BlitSurface(&*_surface,&src,target,&dst);
        }
    }
    void blit(SDL_Surface *target, SDL_Point pos) const {
        if (_surface) {
            SDL_Rect dst = {pos.x,pos.y,width(),height()};
            SDL_BlitSurface(&*_surface,nullptr,target,&dst);
        }
    }

    void blit_scaled(SDL_Surface *target, SDL_Rect src, SDL_Rect dst) {
        if (_surface) SDL_BlitScaled(&*_surface,&src,target,&dst);
    }
    void blit_scaled(SDL_Surface *target, SDL_Rect dst) {
        if (_surface) SDL_BlitScaled(&*_surface,nullptr,target,&dst);
    }

#ifdef SDW_OPENGL
    GLuint glTexture(bool mipmap = false) {
        GLuint texture = 0;
        if ((bool)_surface) {
            GLenum texture_format, internal_format, tex_type;
            if (_surface->format->BytesPerPixel == 4) {
                if (_surface->format->Rmask == 0x000000ff) {
                    texture_format = GL_RGBA; // This is used on Linux for 32 bit
                    tex_type = GL_UNSIGNED_INT_8_8_8_8_REV;
                } else {
                    texture_format = GL_BGRA;
                    tex_type = GL_UNSIGNED_INT_8_8_8_8;
                }
                internal_format = GL_RGBA8;
            } else {
                if (_surface->format->Rmask == 0x000000ff) {
                    texture_format = GL_RGB; // This is used on Linux for 24 bit
                    tex_type = GL_UNSIGNED_BYTE;
                } else {
                    texture_format = GL_BGR;
                    tex_type = GL_UNSIGNED_BYTE;
                }
                internal_format = GL_RGB8;
            }

            int alignment = 8;
            while (_surface->pitch%alignment) alignment>>=1; // x%1==0 for any x
            int expected_pitch = (_surface->w*_surface->format->BytesPerPixel+alignment-1)/alignment*alignment;
            glPixelStorei(GL_UNPACK_ALIGNMENT,alignment);
            if (_surface->pitch-expected_pitch>=alignment) // Alignment alone wont't solve it now
                glPixelStorei(GL_UNPACK_ROW_LENGTH,_surface->pitch/_surface->format->BytesPerPixel);
            else glPixelStorei(GL_UNPACK_ROW_LENGTH,0);

            glGenTextures(1, &texture);
            glBindTexture(GL_TEXTURE_2D, texture);

            glTexImage2D(GL_TEXTURE_2D, 0, internal_format, _surface->w, _surface->h, 0, texture_format, tex_type, _surface->pixels);
            if (mipmap) {
                glGenerateMipmap(GL_TEXTURE_2D);
                glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
            } else {
                glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_BASE_LEVEL, 0);
                glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_LEVEL, 0);
                glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
            }
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

            glPixelStorei(GL_UNPACK_ALIGNMENT,4);
            glPixelStorei(GL_UNPACK_ROW_LENGTH,0);
        }
        return texture;
    }
#endif

protected:
    std::shared_ptr<SDL_Surface> _surface;
};

class Texture {
public:
    Texture() : _renderer(nullptr), _texture(nullptr) {};
    Texture(SDL_Renderer *r, SDL_Texture *tex) throw(Error) :
        _renderer(r), _texture(tex,SDL_DestroyTexture) {Assert((bool)_texture);}
    Texture(SDL_Renderer *r, const char *file) throw(Error) :
        _renderer(r), _texture(IMG_LoadTexture(r, file),SDL_DestroyTexture) {Assert((bool)_texture);}
    Texture(SDL_Renderer *r, const std::string &file) throw(Error) :
        _renderer(r), _texture(IMG_LoadTexture(r, file.c_str()),SDL_DestroyTexture) {Assert((bool)_texture);}
    Texture(SDL_Renderer *r, SDL_Surface *surf) throw(Error) :
        _renderer(r), _texture(SDL_CreateTextureFromSurface(r, surf),SDL_DestroyTexture) {Assert((bool)_texture);}

    operator SDL_Texture*() const {return &*_texture;}

    SDL_Rect rect() const {
        int w,h;
        SDL_QueryTexture(&*_texture,NULL,NULL,&w,&h);
        return {0,0,w,h};
    }
    int width() const {
        if (!_texture) return 0;
        int w;
        SDL_QueryTexture(&*_texture, NULL, NULL, &w, NULL);
        return w;
    }
    int height() const {
        if (!_texture) return 0;
        int h;
        SDL_QueryTexture(&*_texture, NULL, NULL, NULL, &h);
        return h;
    }

    void render(SDL_Rect src, SDL_Rect dst) const {
        SDL_RenderCopy(_renderer, &*_texture, &src, &dst);
    }
    void render(SDL_Rect dst) const {
        SDL_RenderCopy(_renderer, &*_texture, nullptr, &dst);
    }
    void render(SDL_Rect src, SDL_Point pos) const {
        SDL_Rect dst = {pos.x,pos.y,width(),height()};
        SDL_RenderCopy(_renderer, &*_texture, &src, &dst);
    }
    void render(SDL_Point pos) const {
        SDL_Rect dst = {pos.x,pos.y,width(),height()};
        SDL_RenderCopy(_renderer, &*_texture, nullptr, &dst);
    }

    #ifdef SDW_OPENGL
    void gl_bind(float &w, float &h) const {SDL_GL_BindTexture(&*_texture,&w,&h);}
    void gl_unbind() const {SDL_GL_UnbindTexture(&*_texture);}

    void gl_render(SDL_Rect dst) const {
        float w,h;
        glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
        gl_bind(w,h);
        glBegin(GL_TRIANGLE_STRIP);
            glColor3f(1.f,1.f,1.f);
            glTexCoord2f(0.f,0.f); glVertex2f(dst.x,dst.y);
            glTexCoord2f(0.f,h); glVertex2f(dst.x,dst.y+dst.h);
            glTexCoord2f(w,0.f); glVertex2f(dst.x+dst.w,dst.y);
            glTexCoord2f(w,h); glVertex2f(dst.x+dst.w,dst.y+dst.h);
        glEnd();
        gl_unbind();
    }
    void gl_render(SDL_Point dst) const {gl_render(Rect(dst.x,dst.y,width(),height()));}
    void gl_render(SDL_Rect src, SDL_Rect dst) const {
        float w,h;
        glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
        gl_bind(w,h);
        glBegin(GL_TRIANGLE_STRIP);
            glColor3f(1.f,1.f,1.f);
            glTexCoord2f(src.x*w/width(),src.y*h/height()); glVertex2f(dst.x,dst.y);
            glTexCoord2f(src.x*w/width(),(src.y+src.h)*h/height()); glVertex2f(dst.x,dst.y+dst.h);
            glTexCoord2f((src.x+src.w)*w/width(),src.y*h/height()); glVertex2f(dst.x+dst.w,dst.y);
            glTexCoord2f((src.x+src.w)*w/width(),(src.y+src.h)*h/height()); glVertex2f(dst.x+dst.w,dst.y+dst.h);
        glEnd();
        gl_unbind();
    }
    void gl_render(SDL_Rect src, SDL_Point dst) const {gl_render(src,Rect(dst.x,dst.y,width(),height()));}
    void gl_render_rst(SDL_Point center, float rotation, float scalex = 1.f, float scaley = 1.f, SDL_Color tint = {255,255,255,255}) const {
        float w,h;
        float sinr = std::sin(rotation);
        float cosr = std::cos(rotation);
        float xx = cosr*scalex*width()*.5f;
        float yx = sinr*scaley*height()*.5f;
        float xy = -sinr*scalex*width()*.5f;
        float yy = cosr*scaley*height()*.5f;
        glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
        gl_bind(w,h);
        glBegin(GL_TRIANGLE_STRIP);
            glColor4f(tint.r/255.f,tint.g/255.f,tint.b/255.f,tint.a/255.f);
            glTexCoord2f(0.f,0.f); glVertex2f(center.x-xx-yx,center.y-xy-yy);
            glTexCoord2f(0.f,h); glVertex2f(center.x-xx+yx,center.y-xy+yy);
            glTexCoord2f(w,0.f); glVertex2f(center.x+xx-yx,center.y+xy-yy);
            glTexCoord2f(w,h); glVertex2f(center.x+xx+yx,center.y+xy+yy);
        glEnd();
        gl_unbind();
    }
    void gl_render_rst(SDL_Rect src, SDL_Point center, float rotation, float scalex = 1.f, float scaley = 1.f, SDL_Color tint = {255,255,255,255}) const {
        float w,h;
        float sinr = std::sin(rotation);
        float cosr = std::cos(rotation);
        float xx = cosr*scalex*width()*.5f;
        float yx = sinr*scaley*height()*.5f;
        float xy = -sinr*scalex*width()*.5f;
        float yy = cosr*scaley*height()*.5f;
        glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
        gl_bind(w,h);
        glBegin(GL_TRIANGLE_STRIP);
            glColor4f(tint.r/255.f,tint.g/255.f,tint.b/255.f,tint.a/255.f);
            glTexCoord2f(src.x*w/width(),src.y*h/height()); glVertex2f(center.x-xx-yx,center.y-xy-yy);
            glTexCoord2f(src.x*w/width(),(src.y+src.h)*h/height()); glVertex2f(center.x-xx+yx,center.y-xy+yy);
            glTexCoord2f((src.x+src.w)*w/width(),src.y*h/height()); glVertex2f(center.x+xx-yx,center.y+xy-yy);
            glTexCoord2f((src.x+src.w)*w/width(),(src.y+src.h)*h/height()); glVertex2f(center.x+xx+yx,center.y+xy+yy);
        glEnd();
        gl_unbind();
    }
    #endif

protected:
    SDL_Renderer *_renderer;
    std::shared_ptr<SDL_Texture> _texture;
};

// Only handles UTF-8 encoding (which is what SDL uses most of the time anyway).
// If you want weird semi-portable stuff (like Windows' UTF-16), stick your head in a pig.
class Font {
public:
    Font(const std::string &file, int fontsize) throw(Error) :
        _font(TTF_OpenFont(file.c_str(),fontsize),TTF_CloseFont) {Assert((bool)_font);}

    operator TTF_Font*() const {return &*_font;}

    void setStyle(int style = TTF_STYLE_NORMAL) {TTF_SetFontStyle(&*_font,style);}
    int getStyle() {return TTF_GetFontStyle(&*_font);}
    void setOutline(int outline = 0) {TTF_SetFontOutline(&*_font,outline);}
    int getOutline() {return TTF_GetFontOutline(&*_font);}
    void setKerning(bool allowed = true) {TTF_SetFontKerning(&*_font,allowed);}
    bool getKerning() {return TTF_GetFontKerning(&*_font);}
    void setHinting(int hinting = TTF_HINTING_NORMAL) {TTF_SetFontHinting(&*_font,hinting);}
    int getHinting() {return TTF_GetFontHinting(&*_font);}

    int height() {return TTF_FontHeight(&*_font);}
    int ascent() {return TTF_FontAscent(&*_font);}
    int descent() {return TTF_FontDescent(&*_font);}
    int lineskip() {return TTF_FontLineSkip(&*_font);}

    int length(const char* text)
        {int w; TTF_SizeUTF8(&*_font,text,&w,nullptr); return w;}
    int length(const std::string &text) {return length(text.c_str());}

    Surface write_solid(const char *text, SDL_Color color = {255, 255, 255, 255}) {
        if (text == nullptr || *text == 0) return Surface(0,height()); else return TTF_RenderUTF8_Solid(&*_font, text, color);}
    Surface write_solid(const std::string &text, SDL_Color color = {255, 255, 255, 255}) {
        if (text.empty()) return Surface(); else return TTF_RenderUTF8_Solid(&*_font, text.c_str(), color);}
    Texture write_solid(SDL_Renderer *r, const char *text, SDL_Color color = {255, 255, 255, 255}) {
        return Texture(r,write_solid(text, color));}
    Texture write_solid(SDL_Renderer *r, const std::string &text, SDL_Color color = {255, 255, 255, 255}) {
        return Texture(r,write_solid(text, color));}

    Surface write(const char *text, SDL_Color color = {255, 255, 255, 255}) {
        if (text == nullptr || *text == 0) return Surface(0,height()); else return TTF_RenderUTF8_Blended(&*_font, text, color);}
    Surface write(const std::string &text, SDL_Color color = {255, 255, 255, 255}) {
        if (text.empty()) return Surface(0,height()); else return TTF_RenderUTF8_Blended(&*_font, text.c_str(), color);}
    Texture write(SDL_Renderer *r, const char *text, SDL_Color color = {255, 255, 255, 255}) {
        return Texture(r,write(text, color));}
    Texture write(SDL_Renderer *r, const std::string &text, SDL_Color color = {255, 255, 255, 255}) {
        return Texture(r,write(text, color));}

    Surface write(const char *text, SDL_Color fg, SDL_Color bg) {
        if (text == nullptr || *text == 0) return Surface(0,height()); else return TTF_RenderUTF8_Shaded(&*_font, text, fg, bg);}
    Surface write(const std::string &text, SDL_Color fg, SDL_Color bg) {
        if (text.empty()) return Surface(); else return TTF_RenderUTF8_Shaded(&*_font, text.c_str(), fg, bg);}
    Texture write(SDL_Renderer *r, const char *text, SDL_Color fg, SDL_Color bg) {
        return Texture(r,write(text, fg, bg));}
    Texture write(SDL_Renderer *r, const std::string &text, SDL_Color fg, SDL_Color bg) {
        return Texture(r,write(text, fg, bg));}

protected:
    std::shared_ptr<TTF_Font> _font;
};

} // namespace sdw

#endif // SDW_HPP
